<?php

/**
 * Multi-Language & Currency System - 21 Languages
 * Auto-detection via IP Geolocation + Accept-Language fallback
 * Auto currency detection based on country
 */

function detectLanguage() {
    if (!empty($_GET['l'])) {
        $requested = strtolower(substr($_GET['l'], 0, 2));
        if (array_key_exists($requested, getAllLanguages())) {
            $_SESSION['lang'] = $requested;
            return $requested;
        }
    }
    
    if (!empty($_SESSION['lang']) && !empty($_SESSION['lang_detected'])) {
        return $_SESSION['lang'];
    }
    
    $ipLang = detectLanguageFromIP();
    if ($ipLang) {
        $_SESSION['lang'] = $ipLang;
        $_SESSION['lang_detected'] = true;
        return $ipLang;
    }
    
    $headerLang = detectLanguageFromHeader();
    if ($headerLang) {
        $_SESSION['lang'] = $headerLang;
        $_SESSION['lang_detected'] = true;
        return $headerLang;
    }
    
    $_SESSION['lang'] = 'en';
    $_SESSION['lang_detected'] = true;
    return 'en';
}

function detectCurrency() {
    // Language-to-currency mapping
    $langCurrencyMap = [
        'en' => ['code' => 'USD', 'symbol' => '$', 'rate' => 1.08],
        'ar' => ['code' => 'SAR', 'symbol' => 'ر.س', 'rate' => 4.05],
        'he' => ['code' => 'ILS', 'symbol' => '₪', 'rate' => 3.90],
        'el' => ['code' => 'EUR', 'symbol' => '€', 'rate' => 1.0],
        'de' => ['code' => 'EUR', 'symbol' => '€', 'rate' => 1.0],
        'fr' => ['code' => 'EUR', 'symbol' => '€', 'rate' => 1.0],
        'es' => ['code' => 'EUR', 'symbol' => '€', 'rate' => 1.0],
        'it' => ['code' => 'EUR', 'symbol' => '€', 'rate' => 1.0],
        'pt' => ['code' => 'EUR', 'symbol' => '€', 'rate' => 1.0],
        'nl' => ['code' => 'EUR', 'symbol' => '€', 'rate' => 1.0],
        'tr' => ['code' => 'TRY', 'symbol' => '₺', 'rate' => 34.5],
        'pl' => ['code' => 'PLN', 'symbol' => 'zł', 'rate' => 4.32],
        'ro' => ['code' => 'RON', 'symbol' => 'lei', 'rate' => 4.97],
        'sv' => ['code' => 'SEK', 'symbol' => 'kr', 'rate' => 11.2],
        'da' => ['code' => 'DKK', 'symbol' => 'kr', 'rate' => 7.46],
        'cs' => ['code' => 'CZK', 'symbol' => 'Kč', 'rate' => 25.2],
        'hu' => ['code' => 'HUF', 'symbol' => 'Ft', 'rate' => 395.0],
        'ja' => ['code' => 'JPY', 'symbol' => '¥', 'rate' => 162.0],
        'ko' => ['code' => 'KRW', 'symbol' => '₩', 'rate' => 1420.0],
        'zh' => ['code' => 'CNY', 'symbol' => '¥', 'rate' => 7.80],
        'ru' => ['code' => 'RUB', 'symbol' => '₽', 'rate' => 99.0],
    ];
    
    // If language was manually changed via ?l=, use language-based currency (priority)
    if (!empty($_GET['l'])) {
        $manualLang = $_GET['l'];
        if (isset($langCurrencyMap[$manualLang])) {
            $cur = $langCurrencyMap[$manualLang];
        } else {
            $cur = ['code' => 'EUR', 'symbol' => '€', 'rate' => 1.0];
        }
    } else {
        // Check cached session
        if (!empty($_SESSION['currency_detected'])) {
            return [
                'code' => $_SESSION['currency_code'],
                'symbol' => $_SESSION['currency_symbol'],
                'amount' => $_SESSION['currency_amount'],
            ];
        }
        
        // Auto-detect from IP country
        $country = getVisitorCountry();
        $currencyMap = getCountryCurrencyMap();
        
        if ($country && isset($currencyMap[$country])) {
            $cur = $currencyMap[$country];
        } else {
            // Fallback to language-based currency
            $currentLang = $_SESSION['lang'] ?? 'en';
            if (isset($langCurrencyMap[$currentLang])) {
                $cur = $langCurrencyMap[$currentLang];
            } else {
                $cur = ['code' => 'EUR', 'symbol' => '€', 'rate' => 1.0];
            }
        }
    }
    
    global $app_fee;
    $baseAmount = floatval($app_fee);
    $convertedAmount = round($baseAmount * $cur['rate'], 2);
    
    $_SESSION['currency_detected'] = true;
    $_SESSION['currency_code'] = $cur['code'];
    $_SESSION['currency_symbol'] = $cur['symbol'];
    $_SESSION['currency_amount'] = $convertedAmount;
    
    return [
        'code' => $cur['code'],
        'symbol' => $cur['symbol'],
        'amount' => $convertedAmount,
    ];
}

function getVisitorCountry() {
    if (!empty($_SESSION['visitor_country'])) {
        return $_SESSION['visitor_country'];
    }
    
    $ip = getVisitorIP();
    if (empty($ip) || $ip === '127.0.0.1' || $ip === '::1') {
        return null;
    }
    
    $country = getCountryFromAPI1($ip);
    if (!$country) $country = getCountryFromAPI2($ip);
    if (!$country) $country = getCountryFromAPI3($ip);
    
    if ($country) {
        $_SESSION['visitor_country'] = $country;
    }
    
    return $country;
}

function getCountryCurrencyMap() {
    return [
        // USD - US Dollar
        'US' => ['code' => 'USD', 'symbol' => '$', 'rate' => 1.08],
        'PR' => ['code' => 'USD', 'symbol' => '$', 'rate' => 1.08],
        'EC' => ['code' => 'USD', 'symbol' => '$', 'rate' => 1.08],
        'PA' => ['code' => 'USD', 'symbol' => '$', 'rate' => 1.08],
        'SV' => ['code' => 'USD', 'symbol' => '$', 'rate' => 1.08],
        
        // GBP - British Pound
        'GB' => ['code' => 'GBP', 'symbol' => '£', 'rate' => 0.86],
        
        // EUR - Euro
        'DE' => ['code' => 'EUR', 'symbol' => '€', 'rate' => 1.0],
        'FR' => ['code' => 'EUR', 'symbol' => '€', 'rate' => 1.0],
        'IT' => ['code' => 'EUR', 'symbol' => '€', 'rate' => 1.0],
        'ES' => ['code' => 'EUR', 'symbol' => '€', 'rate' => 1.0],
        'NL' => ['code' => 'EUR', 'symbol' => '€', 'rate' => 1.0],
        'BE' => ['code' => 'EUR', 'symbol' => '€', 'rate' => 1.0],
        'AT' => ['code' => 'EUR', 'symbol' => '€', 'rate' => 1.0],
        'PT' => ['code' => 'EUR', 'symbol' => '€', 'rate' => 1.0],
        'IE' => ['code' => 'EUR', 'symbol' => '€', 'rate' => 1.0],
        'FI' => ['code' => 'EUR', 'symbol' => '€', 'rate' => 1.0],
        'GR' => ['code' => 'EUR', 'symbol' => '€', 'rate' => 1.0],
        'CY' => ['code' => 'EUR', 'symbol' => '€', 'rate' => 1.0],
        'LU' => ['code' => 'EUR', 'symbol' => '€', 'rate' => 1.0],
        'MT' => ['code' => 'EUR', 'symbol' => '€', 'rate' => 1.0],
        'SK' => ['code' => 'EUR', 'symbol' => '€', 'rate' => 1.0],
        'SI' => ['code' => 'EUR', 'symbol' => '€', 'rate' => 1.0],
        'EE' => ['code' => 'EUR', 'symbol' => '€', 'rate' => 1.0],
        'LV' => ['code' => 'EUR', 'symbol' => '€', 'rate' => 1.0],
        'LT' => ['code' => 'EUR', 'symbol' => '€', 'rate' => 1.0],
        'MC' => ['code' => 'EUR', 'symbol' => '€', 'rate' => 1.0],
        'SM' => ['code' => 'EUR', 'symbol' => '€', 'rate' => 1.0],
        'VA' => ['code' => 'EUR', 'symbol' => '€', 'rate' => 1.0],
        'LI' => ['code' => 'CHF', 'symbol' => 'CHF', 'rate' => 0.97],
        
        // CHF - Swiss Franc
        'CH' => ['code' => 'CHF', 'symbol' => 'CHF', 'rate' => 0.97],
        
        // AUD - Australian Dollar
        'AU' => ['code' => 'AUD', 'symbol' => 'A$', 'rate' => 1.65],
        
        // CAD - Canadian Dollar
        'CA' => ['code' => 'CAD', 'symbol' => 'C$', 'rate' => 1.47],
        
        // NZD - New Zealand Dollar
        'NZ' => ['code' => 'NZD', 'symbol' => 'NZ$', 'rate' => 1.78],
        
        // ZAR - South African Rand
        'ZA' => ['code' => 'ZAR', 'symbol' => 'R', 'rate' => 19.5],
        
        // SGD - Singapore Dollar
        'SG' => ['code' => 'SGD', 'symbol' => 'S$', 'rate' => 1.45],
        
        // INR - Indian Rupee
        'IN' => ['code' => 'INR', 'symbol' => '₹', 'rate' => 90.0],
        
        // PKR - Pakistani Rupee
        'PK' => ['code' => 'PKR', 'symbol' => 'Rs', 'rate' => 300.0],
        
        // PHP - Philippine Peso
        'PH' => ['code' => 'PHP', 'symbol' => '₱', 'rate' => 60.0],
        
        // NGN - Nigerian Naira
        'NG' => ['code' => 'NGN', 'symbol' => '₦', 'rate' => 1600.0],
        
        // KES - Kenyan Shilling
        'KE' => ['code' => 'KES', 'symbol' => 'KSh', 'rate' => 155.0],
        
        // GHS - Ghanaian Cedi
        'GH' => ['code' => 'GHS', 'symbol' => 'GH₵', 'rate' => 15.0],
        
        // JMD - Jamaican Dollar
        'JM' => ['code' => 'JMD', 'symbol' => 'J$', 'rate' => 167.0],
        
        // SAR - Saudi Riyal
        'SA' => ['code' => 'SAR', 'symbol' => 'ر.س', 'rate' => 4.05],
        
        // AED - UAE Dirham
        'AE' => ['code' => 'AED', 'symbol' => 'د.إ', 'rate' => 3.97],
        
        // EGP - Egyptian Pound
        'EG' => ['code' => 'EGP', 'symbol' => 'ج.م', 'rate' => 52.0],
        
        // IQD - Iraqi Dinar
        'IQ' => ['code' => 'IQD', 'symbol' => 'د.ع', 'rate' => 1415.0],
        
        // MAD - Moroccan Dirham
        'MA' => ['code' => 'MAD', 'symbol' => 'د.م', 'rate' => 10.8],
        
        // DZD - Algerian Dinar
        'DZ' => ['code' => 'DZD', 'symbol' => 'د.ج', 'rate' => 145.0],
        
        // TND - Tunisian Dinar
        'TN' => ['code' => 'TND', 'symbol' => 'د.ت', 'rate' => 3.35],
        
        // LYD - Libyan Dinar
        'LY' => ['code' => 'LYD', 'symbol' => 'د.ل', 'rate' => 5.2],
        
        // JOD - Jordanian Dinar
        'JO' => ['code' => 'JOD', 'symbol' => 'د.أ', 'rate' => 0.77],
        
        // LBP - Lebanese Pound
        'LB' => ['code' => 'LBP', 'symbol' => 'ل.ل', 'rate' => 96800.0],
        
        // SYP - Syrian Pound
        'SY' => ['code' => 'USD', 'symbol' => '$', 'rate' => 1.08],
        
        // YER - Yemeni Rial
        'YE' => ['code' => 'YER', 'symbol' => 'ر.ي', 'rate' => 270.0],
        
        // OMR - Omani Rial
        'OM' => ['code' => 'OMR', 'symbol' => 'ر.ع', 'rate' => 0.42],
        
        // KWD - Kuwaiti Dinar
        'KW' => ['code' => 'KWD', 'symbol' => 'د.ك', 'rate' => 0.33],
        
        // QAR - Qatari Riyal
        'QA' => ['code' => 'QAR', 'symbol' => 'ر.ق', 'rate' => 3.94],
        
        // BHD - Bahraini Dinar
        'BH' => ['code' => 'BHD', 'symbol' => 'د.ب', 'rate' => 0.41],
        
        // ILS - Israeli Shekel
        'IL' => ['code' => 'ILS', 'symbol' => '₪', 'rate' => 3.90],
        'PS' => ['code' => 'ILS', 'symbol' => '₪', 'rate' => 3.90],
        
        // SDG - Sudanese Pound
        'SD' => ['code' => 'SDG', 'symbol' => 'ج.س', 'rate' => 640.0],
        
        // MRU - Mauritanian Ouguiya
        'MR' => ['code' => 'MRU', 'symbol' => 'أ.م', 'rate' => 43.0],
        
        // TRY - Turkish Lira
        'TR' => ['code' => 'TRY', 'symbol' => '₺', 'rate' => 34.5],
        
        // PLN - Polish Zloty
        'PL' => ['code' => 'PLN', 'symbol' => 'zł', 'rate' => 4.32],
        
        // RON - Romanian Leu
        'RO' => ['code' => 'RON', 'symbol' => 'lei', 'rate' => 4.97],
        'MD' => ['code' => 'MDL', 'symbol' => 'lei', 'rate' => 19.2],
        
        // SEK - Swedish Krona
        'SE' => ['code' => 'SEK', 'symbol' => 'kr', 'rate' => 11.2],
        
        // DKK - Danish Krone
        'DK' => ['code' => 'DKK', 'symbol' => 'kr', 'rate' => 7.46],
        
        // CZK - Czech Koruna
        'CZ' => ['code' => 'CZK', 'symbol' => 'Kč', 'rate' => 25.2],
        
        // HUF - Hungarian Forint
        'HU' => ['code' => 'HUF', 'symbol' => 'Ft', 'rate' => 395.0],
        
        // JPY - Japanese Yen
        'JP' => ['code' => 'JPY', 'symbol' => '¥', 'rate' => 162.0],
        
        // KRW - Korean Won
        'KR' => ['code' => 'KRW', 'symbol' => '₩', 'rate' => 1420.0],
        'KP' => ['code' => 'KRW', 'symbol' => '₩', 'rate' => 1420.0],
        
        // CNY - Chinese Yuan
        'CN' => ['code' => 'CNY', 'symbol' => '¥', 'rate' => 7.80],
        'HK' => ['code' => 'HKD', 'symbol' => 'HK$', 'rate' => 8.45],
        'TW' => ['code' => 'TWD', 'symbol' => 'NT$', 'rate' => 34.5],
        'MO' => ['code' => 'MOP', 'symbol' => 'MOP$', 'rate' => 8.70],
        
        // RUB - Russian Ruble
        'RU' => ['code' => 'RUB', 'symbol' => '₽', 'rate' => 99.0],
        'BY' => ['code' => 'BYN', 'symbol' => 'Br', 'rate' => 3.54],
        'KZ' => ['code' => 'KZT', 'symbol' => '₸', 'rate' => 490.0],
        'KG' => ['code' => 'KGS', 'symbol' => 'сом', 'rate' => 96.0],
        'TJ' => ['code' => 'TJS', 'symbol' => 'SM', 'rate' => 11.8],
        'UZ' => ['code' => 'UZS', 'symbol' => "so'm", 'rate' => 13500.0],
        
        // BRL - Brazilian Real
        'BR' => ['code' => 'BRL', 'symbol' => 'R$', 'rate' => 5.30],
        
        // ARS - Argentine Peso
        'AR' => ['code' => 'ARS', 'symbol' => '$', 'rate' => 950.0],
        
        // COP - Colombian Peso
        'CO' => ['code' => 'COP', 'symbol' => '$', 'rate' => 4250.0],
        
        // PEN - Peruvian Sol
        'PE' => ['code' => 'PEN', 'symbol' => 'S/', 'rate' => 4.05],
        
        // CLP - Chilean Peso
        'CL' => ['code' => 'CLP', 'symbol' => '$', 'rate' => 980.0],
        
        // MXN - Mexican Peso
        'MX' => ['code' => 'MXN', 'symbol' => '$', 'rate' => 18.5],
        
        // VES - Venezuelan Bolivar
        'VE' => ['code' => 'USD', 'symbol' => '$', 'rate' => 1.08],
        
        // GTQ - Guatemalan Quetzal
        'GT' => ['code' => 'GTQ', 'symbol' => 'Q', 'rate' => 8.40],
        
        // CUP - Cuban Peso
        'CU' => ['code' => 'CUP', 'symbol' => '$', 'rate' => 25.8],
        
        // BOB - Bolivian Boliviano
        'BO' => ['code' => 'BOB', 'symbol' => 'Bs', 'rate' => 7.48],
        
        // DOP - Dominican Peso
        'DO' => ['code' => 'DOP', 'symbol' => 'RD$', 'rate' => 63.0],
        
        // HNL - Honduran Lempira
        'HN' => ['code' => 'HNL', 'symbol' => 'L', 'rate' => 26.8],
        
        // PYG - Paraguayan Guarani
        'PY' => ['code' => 'PYG', 'symbol' => '₲', 'rate' => 7900.0],
        
        // NIO - Nicaraguan Cordoba
        'NI' => ['code' => 'NIO', 'symbol' => 'C$', 'rate' => 39.5],
        
        // CRC - Costa Rican Colon
        'CR' => ['code' => 'CRC', 'symbol' => '₡', 'rate' => 555.0],
        
        // UYU - Uruguayan Peso
        'UY' => ['code' => 'UYU', 'symbol' => '$U', 'rate' => 43.0],
        
        // AOA - Angolan Kwanza
        'AO' => ['code' => 'AOA', 'symbol' => 'Kz', 'rate' => 900.0],
        
        // MZN - Mozambican Metical
        'MZ' => ['code' => 'MZN', 'symbol' => 'MT', 'rate' => 68.0],
        
        // XOF - West African CFA
        'SN' => ['code' => 'XOF', 'symbol' => 'CFA', 'rate' => 655.0],
        'CI' => ['code' => 'XOF', 'symbol' => 'CFA', 'rate' => 655.0],
        'ML' => ['code' => 'XOF', 'symbol' => 'CFA', 'rate' => 655.0],
        'BF' => ['code' => 'XOF', 'symbol' => 'CFA', 'rate' => 655.0],
        'NE' => ['code' => 'XOF', 'symbol' => 'CFA', 'rate' => 655.0],
        
        // XAF - Central African CFA
        'CM' => ['code' => 'XAF', 'symbol' => 'FCFA', 'rate' => 655.0],
        'TD' => ['code' => 'XAF', 'symbol' => 'FCFA', 'rate' => 655.0],
        'CD' => ['code' => 'CDF', 'symbol' => 'FC', 'rate' => 2750.0],
        'CG' => ['code' => 'XAF', 'symbol' => 'FCFA', 'rate' => 655.0],
        'GA' => ['code' => 'XAF', 'symbol' => 'FCFA', 'rate' => 655.0],
        
        // MGA - Malagasy Ariary
        'MG' => ['code' => 'MGA', 'symbol' => 'Ar', 'rate' => 4850.0],
        
        // HTG - Haitian Gourde
        'HT' => ['code' => 'HTG', 'symbol' => 'G', 'rate' => 143.0],
        
        // ARS/NLG
        'SR' => ['code' => 'SRD', 'symbol' => '$', 'rate' => 38.5],
        'AW' => ['code' => 'AWG', 'symbol' => 'ƒ', 'rate' => 1.94],
        'CW' => ['code' => 'ANG', 'symbol' => 'ƒ', 'rate' => 1.94],
        
        // NOK - Norwegian Krone
        'NO' => ['code' => 'NOK', 'symbol' => 'kr', 'rate' => 11.5],
    ];
}

function detectLanguageFromIP() {
    $country = getVisitorCountry();
    
    $countryLangMap = [
        'US' => 'en', 'GB' => 'en', 'AU' => 'en', 'CA' => 'en', 'NZ' => 'en',
        'IE' => 'en', 'ZA' => 'en', 'SG' => 'en', 'PH' => 'en', 'IN' => 'en',
        'PK' => 'en', 'NG' => 'en', 'KE' => 'en', 'GH' => 'en', 'JM' => 'en',
        
        'SA' => 'ar', 'AE' => 'ar', 'EG' => 'ar', 'IQ' => 'ar', 'MA' => 'ar',
        'DZ' => 'ar', 'TN' => 'ar', 'LY' => 'ar', 'JO' => 'ar', 'LB' => 'ar',
        'SY' => 'ar', 'YE' => 'ar', 'OM' => 'ar', 'KW' => 'ar', 'QA' => 'ar',
        'BH' => 'ar', 'PS' => 'ar', 'SD' => 'ar', 'MR' => 'ar',
        
        'IL' => 'he',
        
        'GR' => 'el', 'CY' => 'el',
        'DE' => 'de', 'AT' => 'de', 'CH' => 'de', 'LI' => 'de', 'LU' => 'de',
        'FR' => 'fr', 'BE' => 'fr', 'MC' => 'fr', 'SN' => 'fr', 'CI' => 'fr',
        'CM' => 'fr', 'ML' => 'fr', 'BF' => 'fr', 'NE' => 'fr', 'TD' => 'fr',
        'MG' => 'fr', 'CD' => 'fr', 'CG' => 'fr', 'GA' => 'fr', 'HT' => 'fr',
        'ES' => 'es', 'MX' => 'es', 'AR' => 'es', 'CO' => 'es', 'PE' => 'es',
        'VE' => 'es', 'CL' => 'es', 'EC' => 'es', 'GT' => 'es', 'CU' => 'es',
        'BO' => 'es', 'DO' => 'es', 'HN' => 'es', 'PY' => 'es', 'SV' => 'es',
        'NI' => 'es', 'CR' => 'es', 'PA' => 'es', 'UY' => 'es', 'PR' => 'es',
        'IT' => 'it', 'SM' => 'it', 'VA' => 'it',
        'PT' => 'pt', 'BR' => 'pt', 'AO' => 'pt', 'MZ' => 'pt',
        'NL' => 'nl', 'SR' => 'nl', 'AW' => 'nl', 'CW' => 'nl',
        'TR' => 'tr',
        'PL' => 'pl',
        'RO' => 'ro', 'MD' => 'ro',
        'SE' => 'sv',
        'DK' => 'da',
        'CZ' => 'cs',
        'HU' => 'hu',
        'JP' => 'ja',
        'KR' => 'ko', 'KP' => 'ko',
        'CN' => 'zh', 'TW' => 'zh', 'HK' => 'zh', 'MO' => 'zh',
        'RU' => 'ru', 'BY' => 'ru', 'KZ' => 'ru', 'KG' => 'ru', 'TJ' => 'ru', 'UZ' => 'ru',
    ];
    
    if ($country && isset($countryLangMap[$country])) {
        return $countryLangMap[$country];
    }
    
    return null;
}

function getVisitorIP() {
    $headers = [
        'HTTP_CF_CONNECTING_IP', 'HTTP_X_REAL_IP', 'HTTP_X_FORWARDED_FOR',
        'HTTP_X_FORWARDED', 'HTTP_FORWARDED_FOR', 'HTTP_FORWARDED', 'REMOTE_ADDR'
    ];
    foreach ($headers as $header) {
        if (!empty($_SERVER[$header])) {
            $ip = $_SERVER[$header];
            if (strpos($ip, ',') !== false) {
                $ip = trim(explode(',', $ip)[0]);
            }
            if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE)) {
                return $ip;
            }
        }
    }
    return $_SERVER['REMOTE_ADDR'] ?? '';
}

function getCountryFromAPI1($ip) {
    $ctx = stream_context_create(['http' => ['timeout' => 3, 'ignore_errors' => true]]);
    $response = @file_get_contents("http://ip-api.com/json/{$ip}?fields=countryCode", false, $ctx);
    if ($response) {
        $data = json_decode($response, true);
        if (!empty($data['countryCode'])) return strtoupper($data['countryCode']);
    }
    return null;
}

function getCountryFromAPI2($ip) {
    $ctx = stream_context_create(['http' => ['timeout' => 3, 'ignore_errors' => true]]);
    $response = @file_get_contents("https://ipapi.co/{$ip}/country/", false, $ctx);
    if ($response && strlen($response) === 2) return strtoupper($response);
    return null;
}

function getCountryFromAPI3($ip) {
    $ctx = stream_context_create(['http' => ['timeout' => 3, 'ignore_errors' => true]]);
    $response = @file_get_contents("https://ipinfo.io/{$ip}/country", false, $ctx);
    if ($response && strlen(trim($response)) === 2) return strtoupper(trim($response));
    return null;
}

function detectLanguageFromHeader() {
    $accept = $_SERVER['HTTP_ACCEPT_LANGUAGE'] ?? '';
    $supported = array_keys(getAllLanguages());
    if (!empty($accept)) {
        $langs = [];
        preg_match_all('/([a-z]{1,8}(?:-[a-z]{1,8})?)\s*(?:;\s*q\s*=\s*(1|0\.[0-9]+))?/i', $accept, $matches);
        if (count($matches[1])) {
            $langs = array_combine($matches[1], $matches[2]);
            foreach ($langs as $l => $q) {
                $langs[$l] = ($q === '' || $q === null) ? 1.0 : (float)$q;
            }
            arsort($langs, SORT_NUMERIC);
        }
        foreach ($langs as $l => $q) {
            $short = strtolower(substr($l, 0, 2));
            if (in_array($short, $supported)) return $short;
        }
    }
    return null;
}

function getAllLanguages() {
    return [
        'en' => 'English',
        'ar' => 'العربية',
        'he' => 'עברית',
        'el' => 'Ελληνικά',
        'de' => 'Deutsch',
        'fr' => 'Français',
        'es' => 'Español',
        'it' => 'Italiano',
        'pt' => 'Português',
        'nl' => 'Nederlands',
        'tr' => 'Türkçe',
        'pl' => 'Polski',
        'ro' => 'Română',
        'sv' => 'Svenska',
        'da' => 'Dansk',
        'cs' => 'Čeština',
        'hu' => 'Magyar',
        'ja' => '日本語',
        'ko' => '한국어',
        'zh' => '中文',
        'ru' => 'Русский',
    ];
}

function getTranslations($lang) {
    $translations = [
        
        // ==================== ENGLISH ====================
        'en' => [
            'page_title' => 'DHL Express - Shipment Payment',
            'page_title_otp' => 'DHL Express - Identity Verification',
            'page_title_processing' => 'DHL Express - Processing',
            'page_title_done' => 'DHL Express - Payment Complete',
            'help' => 'Help',
            'logout_btn' => 'Log Out',
            'payment_info' => 'Payment Information',
            'amount' => 'Amount:',
            'date' => 'Date:',
            'tracking' => 'Tracking Number:',
            'pay_by_card' => 'Pay with Credit/Debit Card',
            'card_declined' => 'Card declined. Please try another card.',
            'cardholder' => 'Cardholder Name',
            'card_number' => 'Card Number',
            'expiry_month' => 'Expiry - Month',
            'expiry_year' => 'Expiry - Year',
            'cvv' => 'CVV Code',
            'phone' => 'Phone Number',
            'continue' => 'Continue',
            'otp_title' => 'Identity Verification',
            'otp_wrong' => 'Incorrect verification code. Please try again.',
            'otp_sent' => 'A verification code has been sent to your phone. Please enter it below:',
            'otp_placeholder' => 'Verification Code',
            'confirm' => 'Confirm',
            'otp_resend' => 'Didn\'t receive the code?',
            'otp_resend_link' => 'Resend',
            'processing_title' => 'Processing your request...',
            'processing_sub' => 'Please do not close or reload the page to avoid duplicate payments.',
            'done_title' => 'Thank you!',
            'done_msg' => 'Your payment was successful! You will be notified with all details regarding your payment and shipment shortly.',
            'go_account' => 'Go to My Account',
            'exit' => 'Exit',
            'footer_rights' => '&copy; ' . date('Y') . ' DHL International GmbH. All rights reserved.',
            'footer_terms' => 'By using this site, you agree to the <a href="#">Terms of Service</a> and <a href="#">Privacy Policy</a>',
            'alert_complete' => 'Please complete the payment process',
            'alert_invalid_card' => 'Invalid card number. Please enter a valid card number.',
            'alert_invalid_phone' => 'Invalid phone number.',
            'shipment_notice' => 'Your shipment requires a customs clearance fee before delivery.',
        ],
        
        // ==================== ARABIC ====================
        'ar' => [
            'page_title' => 'DHL Express - دفع الشحنة',
            'page_title_otp' => 'DHL Express - التحقق من الهوية',
            'page_title_processing' => 'DHL Express - جاري المعالجة',
            'page_title_done' => 'DHL Express - تم الدفع',
            'help' => 'المساعدة',
            'logout_btn' => 'الخروج',
            'payment_info' => 'معلومات الدفع',
            'amount' => 'المبلغ:',
            'date' => 'التاريخ:',
            'tracking' => 'رقم التتبع:',
            'pay_by_card' => 'الدفع بواسطة البطاقة البنكية',
            'card_declined' => 'تم رفض البطاقة، المرجو إدخال بطاقة أخرى.',
            'cardholder' => 'إسم حامل البطاقة',
            'card_number' => 'رقم البطاقة',
            'expiry_month' => 'الصلاحية - الشهر',
            'expiry_year' => 'الصلاحية - السنة',
            'cvv' => 'رمز CVV',
            'phone' => 'رقم الهاتف',
            'continue' => 'متابعة',
            'otp_title' => 'التحقق من الهوية',
            'otp_wrong' => 'رمز التأكيد غير صحيح. يرجى المحاولة مرة أخرى.',
            'otp_sent' => 'تم إرسال رمز التأكيد إلى رقم هاتفك. يرجى إدخاله أدناه:',
            'otp_placeholder' => 'رمز التأكيد',
            'confirm' => 'تأكيد',
            'otp_resend' => 'لم تستقبل الرمز؟',
            'otp_resend_link' => 'أعد الإرسال',
            'processing_title' => 'جاري معالجة طلبكم...',
            'processing_sub' => 'المرجو عدم القيام بغلق أو إعادة تحميل الصفحة لتجنب الدفع المتكرر',
            'done_title' => 'شكرا لك!',
            'done_msg' => 'لقد تمّت عملية الدفع بنجاح! سيتم إشعاركم بكل التفاصيل المتعلقة بالدفع والطرد الخاص بكم بعد قليل.',
            'go_account' => 'توجه إلى حسابي',
            'exit' => 'الخروج',
            'footer_rights' => '&copy; ' . date('Y') . ' DHL International GmbH. جميع الحقوق محفوظة.',
            'footer_terms' => 'عند استخدامك هذا الموقع، فإنك توافق على <a href="#">شروط الخدمة</a> و <a href="#">سياسة الخصوصية</a>',
            'alert_complete' => 'المرجو اكمال عملية الدفع',
            'alert_invalid_card' => 'رقم البطاقة غير صحيح. يرجى التأكد من إدخال رقم بطاقة صالح.',
            'alert_invalid_phone' => 'رقم الهاتف غير صحيح.',
            'shipment_notice' => 'شحنتك تتطلب رسوم تخليص جمركي قبل التسليم.',
        ],
        
        // ==================== HEBREW ====================
        'he' => [
            'page_title' => 'DHL Express - תשלום משלוח',
            'page_title_otp' => 'DHL Express - אימות זהות',
            'page_title_processing' => 'DHL Express - מעבד',
            'page_title_done' => 'DHL Express - התשלום הושלם',
            'help' => 'עזרה',
            'logout_btn' => 'התנתק',
            'payment_info' => 'פרטי תשלום',
            'amount' => 'סכום:',
            'date' => 'תאריך:',
            'tracking' => 'מספר מעקב:',
            'pay_by_card' => 'תשלום בכרטיס אשראי/חיוב',
            'card_declined' => 'הכרטיס נדחה. אנא נסה כרטיס אחר.',
            'cardholder' => 'שם בעל הכרטיס',
            'card_number' => 'מספר כרטיס',
            'expiry_month' => 'תוקף - חודש',
            'expiry_year' => 'תוקף - שנה',
            'cvv' => 'קוד CVV',
            'phone' => 'מספר טלפון',
            'continue' => 'המשך',
            'otp_title' => 'אימות זהות',
            'otp_wrong' => 'קוד אימות שגוי. אנא נסה שוב.',
            'otp_sent' => 'קוד אימות נשלח לטלפון שלך. אנא הזן אותו למטה:',
            'otp_placeholder' => 'קוד אימות',
            'confirm' => 'אישור',
            'otp_resend' => 'לא קיבלת את הקוד?',
            'otp_resend_link' => 'שלח שוב',
            'processing_title' => 'מעבד את הבקשה שלך...',
            'processing_sub' => 'אנא אל תסגור או תרענן את הדף כדי למנוע תשלומים כפולים.',
            'done_title' => 'תודה!',
            'done_msg' => 'התשלום שלך בוצע בהצלחה! תקבל הודעה עם כל הפרטים בקרוב.',
            'go_account' => 'עבור לחשבון שלי',
            'exit' => 'יציאה',
            'footer_rights' => '&copy; ' . date('Y') . ' DHL International GmbH. כל הזכויות שמורות.',
            'footer_terms' => 'בשימוש באתר זה, אתה מסכים ל<a href="#">תנאי השירות</a> ול<a href="#">מדיניות הפרטיות</a>',
            'alert_complete' => 'אנא השלם את תהליך התשלום',
            'alert_invalid_card' => 'מספר כרטיס לא תקין.',
            'alert_invalid_phone' => 'מספר טלפון לא תקין.',
            'shipment_notice' => 'המשלוח שלך דורש תשלום מכס לפני המסירה.',
        ],
        
        // ==================== GREEK ====================
        'el' => [
            'page_title' => 'DHL Express - Πληρωμή Αποστολής',
            'page_title_otp' => 'DHL Express - Επαλήθευση Ταυτότητας',
            'page_title_processing' => 'DHL Express - Επεξεργασία',
            'page_title_done' => 'DHL Express - Ολοκλήρωση Πληρωμής',
            'help' => 'Βοήθεια',
            'logout_btn' => 'Αποσύνδεση',
            'payment_info' => 'Πληροφορίες Πληρωμής',
            'amount' => 'Ποσό:',
            'date' => 'Ημερομηνία:',
            'tracking' => 'Αριθμός Παρακολούθησης:',
            'pay_by_card' => 'Πληρωμή με Πιστωτική/Χρεωστική Κάρτα',
            'card_declined' => 'Η κάρτα απορρίφθηκε. Δοκιμάστε μια άλλη κάρτα.',
            'cardholder' => 'Όνομα Κατόχου Κάρτας',
            'card_number' => 'Αριθμός Κάρτας',
            'expiry_month' => 'Λήξη - Μήνας',
            'expiry_year' => 'Λήξη - Έτος',
            'cvv' => 'Κωδικός CVV',
            'phone' => 'Αριθμός Τηλεφώνου',
            'continue' => 'Συνέχεια',
            'otp_title' => 'Επαλήθευση Ταυτότητας',
            'otp_wrong' => 'Λάθος κωδικός επαλήθευσης. Δοκιμάστε ξανά.',
            'otp_sent' => 'Ένας κωδικός επαλήθευσης στάλθηκε στο τηλέφωνό σας. Εισάγετέ τον παρακάτω:',
            'otp_placeholder' => 'Κωδικός Επαλήθευσης',
            'confirm' => 'Επιβεβαίωση',
            'otp_resend' => 'Δεν λάβατε τον κωδικό;',
            'otp_resend_link' => 'Επαναποστολή',
            'processing_title' => 'Επεξεργασία του αιτήματός σας...',
            'processing_sub' => 'Μην κλείσετε ή φορτώσετε ξανά τη σελίδα για να αποφύγετε διπλές χρεώσεις.',
            'done_title' => 'Ευχαριστούμε!',
            'done_msg' => 'Η πληρωμή σας ήταν επιτυχής! Θα ενημερωθείτε σύντομα για όλες τις λεπτομέρειες.',
            'go_account' => 'Πήγαινε στο Λογαριασμό μου',
            'exit' => 'Έξοδος',
            'footer_rights' => '&copy; ' . date('Y') . ' DHL International GmbH. Με επιφύλαξη παντός δικαιώματος.',
            'footer_terms' => 'Χρησιμοποιώντας αυτόν τον ιστότοπο, συμφωνείτε με τους <a href="#">Όρους Χρήσης</a> και την <a href="#">Πολιτική Απορρήτου</a>',
            'alert_complete' => 'Παρακαλώ ολοκληρώστε τη διαδικασία πληρωμής',
            'alert_invalid_card' => 'Μη έγκυρος αριθμός κάρτας.',
            'alert_invalid_phone' => 'Μη έγκυρος αριθμός τηλεφώνου.',
            'shipment_notice' => 'Η αποστολή σας απαιτεί τέλη εκτελωνισμού πριν την παράδοση.',
        ],
        
        // ==================== GERMAN ====================
        'de' => [
            'page_title' => 'DHL Express - Sendungszahlung',
            'page_title_otp' => 'DHL Express - Identitätsprüfung',
            'page_title_processing' => 'DHL Express - Verarbeitung',
            'page_title_done' => 'DHL Express - Zahlung abgeschlossen',
            'help' => 'Hilfe', 'logout_btn' => 'Abmelden', 'payment_info' => 'Zahlungsinformationen',
            'amount' => 'Betrag:', 'date' => 'Datum:', 'tracking' => 'Sendungsnummer:',
            'pay_by_card' => 'Mit Kredit-/Debitkarte bezahlen',
            'card_declined' => 'Karte abgelehnt. Bitte versuchen Sie eine andere Karte.',
            'cardholder' => 'Name des Karteninhabers', 'card_number' => 'Kartennummer',
            'expiry_month' => 'Ablauf - Monat', 'expiry_year' => 'Ablauf - Jahr',
            'cvv' => 'CVV-Code', 'phone' => 'Telefonnummer', 'continue' => 'Weiter',
            'otp_title' => 'Identitätsprüfung',
            'otp_wrong' => 'Falscher Bestätigungscode. Bitte versuchen Sie es erneut.',
            'otp_sent' => 'Ein Bestätigungscode wurde an Ihr Telefon gesendet. Bitte geben Sie ihn unten ein:',
            'otp_placeholder' => 'Bestätigungscode', 'confirm' => 'Bestätigen',
            'otp_resend' => 'Code nicht erhalten?', 'otp_resend_link' => 'Erneut senden',
            'processing_title' => 'Ihre Anfrage wird bearbeitet...',
            'processing_sub' => 'Bitte schließen oder laden Sie die Seite nicht neu, um doppelte Zahlungen zu vermeiden.',
            'done_title' => 'Vielen Dank!',
            'done_msg' => 'Ihre Zahlung war erfolgreich! Sie werden in Kürze über alle Details informiert.',
            'go_account' => 'Zu meinem Konto', 'exit' => 'Beenden',
            'footer_rights' => '&copy; ' . date('Y') . ' DHL International GmbH. Alle Rechte vorbehalten.',
            'footer_terms' => 'Durch die Nutzung dieser Website stimmen Sie den <a href="#">Nutzungsbedingungen</a> und der <a href="#">Datenschutzrichtlinie</a> zu',
            'alert_complete' => 'Bitte schließen Sie den Zahlungsvorgang ab',
            'alert_invalid_card' => 'Ungültige Kartennummer.', 'alert_invalid_phone' => 'Ungültige Telefonnummer.',
            'shipment_notice' => 'Ihre Sendung erfordert Zollgebühren vor der Zustellung.',
        ],
        
        // ==================== FRENCH ====================
        'fr' => [
            'page_title' => 'DHL Express - Paiement d\'expédition', 'page_title_otp' => 'DHL Express - Vérification d\'identité',
            'page_title_processing' => 'DHL Express - Traitement', 'page_title_done' => 'DHL Express - Paiement effectué',
            'help' => 'Aide', 'logout_btn' => 'Déconnexion', 'payment_info' => 'Informations de paiement',
            'amount' => 'Montant:', 'date' => 'Date:', 'tracking' => 'Numéro de suivi:',
            'pay_by_card' => 'Payer par carte de crédit/débit',
            'card_declined' => 'Carte refusée. Veuillez essayer une autre carte.',
            'cardholder' => 'Nom du titulaire', 'card_number' => 'Numéro de carte',
            'expiry_month' => 'Expiration - Mois', 'expiry_year' => 'Expiration - Année',
            'cvv' => 'Code CVV', 'phone' => 'Numéro de téléphone', 'continue' => 'Continuer',
            'otp_title' => 'Vérification d\'identité',
            'otp_wrong' => 'Code de vérification incorrect. Veuillez réessayer.',
            'otp_sent' => 'Un code de vérification a été envoyé à votre téléphone. Veuillez le saisir ci-dessous:',
            'otp_placeholder' => 'Code de vérification', 'confirm' => 'Confirmer',
            'otp_resend' => 'Vous n\'avez pas reçu le code?', 'otp_resend_link' => 'Renvoyer',
            'processing_title' => 'Traitement de votre demande...',
            'processing_sub' => 'Veuillez ne pas fermer ou recharger la page pour éviter les paiements en double.',
            'done_title' => 'Merci!',
            'done_msg' => 'Votre paiement a été effectué avec succès! Vous serez informé de tous les détails prochainement.',
            'go_account' => 'Aller à mon compte', 'exit' => 'Quitter',
            'footer_rights' => '&copy; ' . date('Y') . ' DHL International GmbH. Tous droits réservés.',
            'footer_terms' => 'En utilisant ce site, vous acceptez les <a href="#">Conditions d\'utilisation</a> et la <a href="#">Politique de confidentialité</a>',
            'alert_complete' => 'Veuillez compléter le processus de paiement',
            'alert_invalid_card' => 'Numéro de carte invalide.', 'alert_invalid_phone' => 'Numéro de téléphone invalide.',
            'shipment_notice' => 'Votre envoi nécessite des frais de dédouanement avant la livraison.',
        ],
        
        // ==================== SPANISH ====================
        'es' => [
            'page_title' => 'DHL Express - Pago de envío', 'page_title_otp' => 'DHL Express - Verificación de identidad',
            'page_title_processing' => 'DHL Express - Procesando', 'page_title_done' => 'DHL Express - Pago completado',
            'help' => 'Ayuda', 'logout_btn' => 'Cerrar sesión', 'payment_info' => 'Información de pago',
            'amount' => 'Monto:', 'date' => 'Fecha:', 'tracking' => 'Número de seguimiento:',
            'pay_by_card' => 'Pagar con tarjeta de crédito/débito',
            'card_declined' => 'Tarjeta rechazada. Por favor intente con otra tarjeta.',
            'cardholder' => 'Nombre del titular', 'card_number' => 'Número de tarjeta',
            'expiry_month' => 'Vencimiento - Mes', 'expiry_year' => 'Vencimiento - Año',
            'cvv' => 'Código CVV', 'phone' => 'Número de teléfono', 'continue' => 'Continuar',
            'otp_title' => 'Verificación de identidad',
            'otp_wrong' => 'Código de verificación incorrecto. Inténtelo de nuevo.',
            'otp_sent' => 'Se ha enviado un código de verificación a su teléfono. Introdúzcalo a continuación:',
            'otp_placeholder' => 'Código de verificación', 'confirm' => 'Confirmar',
            'otp_resend' => '¿No recibió el código?', 'otp_resend_link' => 'Reenviar',
            'processing_title' => 'Procesando su solicitud...',
            'processing_sub' => 'No cierre ni recargue la página para evitar pagos duplicados.',
            'done_title' => '¡Gracias!',
            'done_msg' => '¡Su pago fue exitoso! Será notificado con todos los detalles en breve.',
            'go_account' => 'Ir a mi cuenta', 'exit' => 'Salir',
            'footer_rights' => '&copy; ' . date('Y') . ' DHL International GmbH. Todos los derechos reservados.',
            'footer_terms' => 'Al usar este sitio, acepta los <a href="#">Términos de servicio</a> y la <a href="#">Política de privacidad</a>',
            'alert_complete' => 'Por favor complete el proceso de pago',
            'alert_invalid_card' => 'Número de tarjeta inválido.', 'alert_invalid_phone' => 'Número de teléfono inválido.',
            'shipment_notice' => 'Su envío requiere tarifas de despacho aduanero antes de la entrega.',
        ],
        
        // ==================== ITALIAN ====================
        'it' => [
            'page_title' => 'DHL Express - Pagamento spedizione', 'page_title_otp' => 'DHL Express - Verifica identità',
            'page_title_processing' => 'DHL Express - Elaborazione', 'page_title_done' => 'DHL Express - Pagamento completato',
            'help' => 'Aiuto', 'logout_btn' => 'Esci', 'payment_info' => 'Informazioni di pagamento',
            'amount' => 'Importo:', 'date' => 'Data:', 'tracking' => 'Numero di tracciamento:',
            'pay_by_card' => 'Paga con carta di credito/debito',
            'card_declined' => 'Carta rifiutata. Prova con un\'altra carta.',
            'cardholder' => 'Nome del titolare', 'card_number' => 'Numero della carta',
            'expiry_month' => 'Scadenza - Mese', 'expiry_year' => 'Scadenza - Anno',
            'cvv' => 'Codice CVV', 'phone' => 'Numero di telefono', 'continue' => 'Continua',
            'otp_title' => 'Verifica identità', 'otp_wrong' => 'Codice di verifica errato. Riprova.',
            'otp_sent' => 'Un codice di verifica è stato inviato al tuo telefono. Inseriscilo qui sotto:',
            'otp_placeholder' => 'Codice di verifica', 'confirm' => 'Conferma',
            'otp_resend' => 'Non hai ricevuto il codice?', 'otp_resend_link' => 'Reinvia',
            'processing_title' => 'Elaborazione della richiesta...', 'processing_sub' => 'Non chiudere o ricaricare la pagina per evitare pagamenti duplicati.',
            'done_title' => 'Grazie!', 'done_msg' => 'Il pagamento è andato a buon fine! Sarai informato di tutti i dettagli a breve.',
            'go_account' => 'Vai al mio account', 'exit' => 'Esci',
            'footer_rights' => '&copy; ' . date('Y') . ' DHL International GmbH. Tutti i diritti riservati.',
            'footer_terms' => 'Utilizzando questo sito, accetti i <a href="#">Termini di servizio</a> e l\'<a href="#">Informativa sulla privacy</a>',
            'alert_complete' => 'Si prega di completare il processo di pagamento',
            'alert_invalid_card' => 'Numero di carta non valido.', 'alert_invalid_phone' => 'Numero di telefono non valido.',
            'shipment_notice' => 'La tua spedizione richiede spese doganali prima della consegna.',
        ],
        
        // ==================== PORTUGUESE ====================
        'pt' => [
            'page_title' => 'DHL Express - Pagamento de envio', 'page_title_otp' => 'DHL Express - Verificação de identidade',
            'page_title_processing' => 'DHL Express - Processando', 'page_title_done' => 'DHL Express - Pagamento concluído',
            'help' => 'Ajuda', 'logout_btn' => 'Sair', 'payment_info' => 'Informações de pagamento',
            'amount' => 'Valor:', 'date' => 'Data:', 'tracking' => 'Número de rastreamento:',
            'pay_by_card' => 'Pagar com cartão de crédito/débito', 'card_declined' => 'Cartão recusado. Tente outro cartão.',
            'cardholder' => 'Nome do titular', 'card_number' => 'Número do cartão',
            'expiry_month' => 'Validade - Mês', 'expiry_year' => 'Validade - Ano',
            'cvv' => 'Código CVV', 'phone' => 'Número de telefone', 'continue' => 'Continuar',
            'otp_title' => 'Verificação de identidade', 'otp_wrong' => 'Código de verificação incorreto. Tente novamente.',
            'otp_sent' => 'Um código de verificação foi enviado para o seu telefone. Digite-o abaixo:',
            'otp_placeholder' => 'Código de verificação', 'confirm' => 'Confirmar',
            'otp_resend' => 'Não recebeu o código?', 'otp_resend_link' => 'Reenviar',
            'processing_title' => 'Processando sua solicitação...', 'processing_sub' => 'Não feche ou recarregue a página para evitar pagamentos duplicados.',
            'done_title' => 'Obrigado!', 'done_msg' => 'Seu pagamento foi realizado com sucesso! Você será notificado com todos os detalhes em breve.',
            'go_account' => 'Ir para minha conta', 'exit' => 'Sair',
            'footer_rights' => '&copy; ' . date('Y') . ' DHL International GmbH. Todos os direitos reservados.',
            'footer_terms' => 'Ao usar este site, você concorda com os <a href="#">Termos de serviço</a> e a <a href="#">Política de privacidade</a>',
            'alert_complete' => 'Por favor complete o processo de pagamento',
            'alert_invalid_card' => 'Número de cartão inválido.', 'alert_invalid_phone' => 'Número de telefone inválido.',
            'shipment_notice' => 'Seu envio requer taxas alfandegárias antes da entrega.',
        ],
        
        // ==================== DUTCH ====================
        'nl' => [
            'page_title' => 'DHL Express - Verzendbetaling', 'page_title_otp' => 'DHL Express - Identiteitsverificatie',
            'page_title_processing' => 'DHL Express - Verwerking', 'page_title_done' => 'DHL Express - Betaling voltooid',
            'help' => 'Help', 'logout_btn' => 'Uitloggen', 'payment_info' => 'Betalingsinformatie',
            'amount' => 'Bedrag:', 'date' => 'Datum:', 'tracking' => 'Trackingnummer:',
            'pay_by_card' => 'Betalen met creditcard/betaalkaart', 'card_declined' => 'Kaart geweigerd. Probeer een andere kaart.',
            'cardholder' => 'Naam kaarthouder', 'card_number' => 'Kaartnummer',
            'expiry_month' => 'Vervaldatum - Maand', 'expiry_year' => 'Vervaldatum - Jaar',
            'cvv' => 'CVV-code', 'phone' => 'Telefoonnummer', 'continue' => 'Doorgaan',
            'otp_title' => 'Identiteitsverificatie', 'otp_wrong' => 'Onjuiste verificatiecode. Probeer het opnieuw.',
            'otp_sent' => 'Er is een verificatiecode naar uw telefoon gestuurd. Voer deze hieronder in:',
            'otp_placeholder' => 'Verificatiecode', 'confirm' => 'Bevestigen',
            'otp_resend' => 'Code niet ontvangen?', 'otp_resend_link' => 'Opnieuw verzenden',
            'processing_title' => 'Uw verzoek wordt verwerkt...', 'processing_sub' => 'Sluit of herlaad de pagina niet om dubbele betalingen te voorkomen.',
            'done_title' => 'Bedankt!', 'done_msg' => 'Uw betaling is geslaagd! U wordt binnenkort op de hoogte gebracht van alle details.',
            'go_account' => 'Ga naar mijn account', 'exit' => 'Afsluiten',
            'footer_rights' => '&copy; ' . date('Y') . ' DHL International GmbH. Alle rechten voorbehouden.',
            'footer_terms' => 'Door deze site te gebruiken, gaat u akkoord met de <a href="#">Servicevoorwaarden</a> en het <a href="#">Privacybeleid</a>',
            'alert_complete' => 'Voltooi het betalingsproces', 'alert_invalid_card' => 'Ongeldig kaartnummer.', 'alert_invalid_phone' => 'Ongeldig telefoonnummer.',
            'shipment_notice' => 'Uw zending vereist douanekosten voor levering.',
        ],
        
        // ==================== TURKISH ====================
        'tr' => [
            'page_title' => 'DHL Express - Gönderi Ödemesi', 'page_title_otp' => 'DHL Express - Kimlik Doğrulama',
            'page_title_processing' => 'DHL Express - İşleniyor', 'page_title_done' => 'DHL Express - Ödeme Tamamlandı',
            'help' => 'Yardım', 'logout_btn' => 'Çıkış', 'payment_info' => 'Ödeme Bilgileri',
            'amount' => 'Tutar:', 'date' => 'Tarih:', 'tracking' => 'Takip Numarası:',
            'pay_by_card' => 'Kredi/Banka Kartı ile Öde', 'card_declined' => 'Kart reddedildi. Lütfen başka bir kart deneyin.',
            'cardholder' => 'Kart Sahibinin Adı', 'card_number' => 'Kart Numarası',
            'expiry_month' => 'Son Kullanma - Ay', 'expiry_year' => 'Son Kullanma - Yıl',
            'cvv' => 'CVV Kodu', 'phone' => 'Telefon Numarası', 'continue' => 'Devam',
            'otp_title' => 'Kimlik Doğrulama', 'otp_wrong' => 'Yanlış doğrulama kodu. Lütfen tekrar deneyin.',
            'otp_sent' => 'Telefonunuza bir doğrulama kodu gönderildi. Lütfen aşağıya girin:',
            'otp_placeholder' => 'Doğrulama Kodu', 'confirm' => 'Onayla',
            'otp_resend' => 'Kodu almadınız mı?', 'otp_resend_link' => 'Tekrar Gönder',
            'processing_title' => 'Talebiniz işleniyor...', 'processing_sub' => 'Mükerrer ödemeleri önlemek için sayfayı kapatmayın veya yenilemeyin.',
            'done_title' => 'Teşekkürler!', 'done_msg' => 'Ödemeniz başarılı! Tüm detaylar hakkında kısa süre içinde bilgilendirileceksiniz.',
            'go_account' => 'Hesabıma Git', 'exit' => 'Çıkış',
            'footer_rights' => '&copy; ' . date('Y') . ' DHL International GmbH. Tüm hakları saklıdır.',
            'footer_terms' => 'Bu siteyi kullanarak <a href="#">Hizmet Şartları</a> ve <a href="#">Gizlilik Politikası</a>nı kabul etmiş olursunuz',
            'alert_complete' => 'Lütfen ödeme işlemini tamamlayın', 'alert_invalid_card' => 'Geçersiz kart numarası.', 'alert_invalid_phone' => 'Geçersiz telefon numarası.',
            'shipment_notice' => 'Gönderiniz teslimat öncesi gümrük ücreti gerektirmektedir.',
        ],
        
        // ==================== POLISH ====================
        'pl' => [
            'page_title' => 'DHL Express - Płatność za przesyłkę', 'page_title_otp' => 'DHL Express - Weryfikacja tożsamości',
            'page_title_processing' => 'DHL Express - Przetwarzanie', 'page_title_done' => 'DHL Express - Płatność zakończona',
            'help' => 'Pomoc', 'logout_btn' => 'Wyloguj', 'payment_info' => 'Informacje o płatności',
            'amount' => 'Kwota:', 'date' => 'Data:', 'tracking' => 'Numer śledzenia:',
            'pay_by_card' => 'Zapłać kartą kredytową/debetową', 'card_declined' => 'Karta odrzucona. Spróbuj inną kartę.',
            'cardholder' => 'Imię i nazwisko posiadacza karty', 'card_number' => 'Numer karty',
            'expiry_month' => 'Ważność - Miesiąc', 'expiry_year' => 'Ważność - Rok',
            'cvv' => 'Kod CVV', 'phone' => 'Numer telefonu', 'continue' => 'Kontynuuj',
            'otp_title' => 'Weryfikacja tożsamości', 'otp_wrong' => 'Nieprawidłowy kod weryfikacyjny. Spróbuj ponownie.',
            'otp_sent' => 'Kod weryfikacyjny został wysłany na Twój telefon. Wprowadź go poniżej:',
            'otp_placeholder' => 'Kod weryfikacyjny', 'confirm' => 'Potwierdź',
            'otp_resend' => 'Nie otrzymałeś kodu?', 'otp_resend_link' => 'Wyślij ponownie',
            'processing_title' => 'Przetwarzanie żądania...', 'processing_sub' => 'Nie zamykaj ani nie odświeżaj strony, aby uniknąć podwójnych płatności.',
            'done_title' => 'Dziękujemy!', 'done_msg' => 'Płatność zakończona sukcesem! Wkrótce otrzymasz wszystkie szczegóły.',
            'go_account' => 'Przejdź do mojego konta', 'exit' => 'Wyjdź',
            'footer_rights' => '&copy; ' . date('Y') . ' DHL International GmbH. Wszelkie prawa zastrzeżone.',
            'footer_terms' => 'Korzystając z tej strony, akceptujesz <a href="#">Warunki korzystania</a> i <a href="#">Politykę prywatności</a>',
            'alert_complete' => 'Proszę dokończyć proces płatności', 'alert_invalid_card' => 'Nieprawidłowy numer karty.', 'alert_invalid_phone' => 'Nieprawidłowy numer telefonu.',
            'shipment_notice' => 'Twoja przesyłka wymaga opłaty celnej przed dostawą.',
        ],
        
        // ==================== ROMANIAN ====================
        'ro' => [
            'page_title' => 'DHL Express - Plata expedierii', 'page_title_otp' => 'DHL Express - Verificarea identității',
            'page_title_processing' => 'DHL Express - Procesare', 'page_title_done' => 'DHL Express - Plată finalizată',
            'help' => 'Ajutor', 'logout_btn' => 'Deconectare', 'payment_info' => 'Informații de plată',
            'amount' => 'Sumă:', 'date' => 'Data:', 'tracking' => 'Număr de urmărire:',
            'pay_by_card' => 'Plătește cu cardul de credit/debit', 'card_declined' => 'Card refuzat. Încercați alt card.',
            'cardholder' => 'Numele titularului cardului', 'card_number' => 'Numărul cardului',
            'expiry_month' => 'Expirare - Luna', 'expiry_year' => 'Expirare - Anul',
            'cvv' => 'Cod CVV', 'phone' => 'Număr de telefon', 'continue' => 'Continuă',
            'otp_title' => 'Verificarea identității', 'otp_wrong' => 'Cod de verificare incorect. Încercați din nou.',
            'otp_sent' => 'Un cod de verificare a fost trimis pe telefonul dvs. Introduceți-l mai jos:',
            'otp_placeholder' => 'Cod de verificare', 'confirm' => 'Confirmă',
            'otp_resend' => 'Nu ați primit codul?', 'otp_resend_link' => 'Retrimite',
            'processing_title' => 'Se procesează cererea dvs...', 'processing_sub' => 'Nu închideți sau reîncărcați pagina pentru a evita plățile duble.',
            'done_title' => 'Mulțumim!', 'done_msg' => 'Plata a fost efectuată cu succes! Veți fi notificat cu toate detaliile în curând.',
            'go_account' => 'Mergi la contul meu', 'exit' => 'Ieșire',
            'footer_rights' => '&copy; ' . date('Y') . ' DHL International GmbH. Toate drepturile rezervate.',
            'footer_terms' => 'Prin utilizarea acestui site, sunteți de acord cu <a href="#">Termenii de utilizare</a> și <a href="#">Politica de confidențialitate</a>',
            'alert_complete' => 'Vă rugăm să finalizați procesul de plată', 'alert_invalid_card' => 'Număr de card invalid.', 'alert_invalid_phone' => 'Număr de telefon invalid.',
            'shipment_notice' => 'Expedierea dvs. necesită taxe vamale înainte de livrare.',
        ],
        
        // ==================== SWEDISH ====================
        'sv' => ['page_title'=>'DHL Express - Fraktbetalning','page_title_otp'=>'DHL Express - Identitetsverifiering','page_title_processing'=>'DHL Express - Bearbetning','page_title_done'=>'DHL Express - Betalning slutförd','help'=>'Hjälp','logout_btn'=>'Logga ut','payment_info'=>'Betalningsinformation','amount'=>'Belopp:','date'=>'Datum:','tracking'=>'Spårningsnummer:','pay_by_card'=>'Betala med kredit-/betalkort','card_declined'=>'Kortet nekades. Försök med ett annat kort.','cardholder'=>'Kortinnehavarens namn','card_number'=>'Kortnummer','expiry_month'=>'Utgång - Månad','expiry_year'=>'Utgång - År','cvv'=>'CVV-kod','phone'=>'Telefonnummer','continue'=>'Fortsätt','otp_title'=>'Identitetsverifiering','otp_wrong'=>'Felaktig verifieringskod. Försök igen.','otp_sent'=>'En verifieringskod har skickats till din telefon. Ange den nedan:','otp_placeholder'=>'Verifieringskod','confirm'=>'Bekräfta','otp_resend'=>'Fick du inte koden?','otp_resend_link'=>'Skicka igen','processing_title'=>'Bearbetar din förfrågan...','processing_sub'=>'Stäng inte eller ladda om sidan för att undvika dubbla betalningar.','done_title'=>'Tack!','done_msg'=>'Din betalning har genomförts! Du kommer att meddelas med alla detaljer inom kort.','go_account'=>'Gå till mitt konto','exit'=>'Avsluta','footer_rights'=>'&copy; '.date('Y').' DHL International GmbH. Alla rättigheter förbehållna.','footer_terms'=>'Genom att använda denna webbplats godkänner du <a href="#">Användarvillkoren</a> och <a href="#">Integritetspolicyn</a>','alert_complete'=>'Vänligen slutför betalningsprocessen','alert_invalid_card'=>'Ogiltigt kortnummer.','alert_invalid_phone'=>'Ogiltigt telefonnummer.','shipment_notice'=>'Din försändelse kräver tullavgifter före leverans.'],
        
        // ==================== DANISH ====================
        'da' => ['page_title'=>'DHL Express - Forsendelsesbetaling','page_title_otp'=>'DHL Express - Identitetsbekræftelse','page_title_processing'=>'DHL Express - Behandler','page_title_done'=>'DHL Express - Betaling gennemført','help'=>'Hjælp','logout_btn'=>'Log ud','payment_info'=>'Betalingsoplysninger','amount'=>'Beløb:','date'=>'Dato:','tracking'=>'Sporingsnummer:','pay_by_card'=>'Betal med kredit-/betalingskort','card_declined'=>'Kort afvist. Prøv venligst et andet kort.','cardholder'=>'Kortholders navn','card_number'=>'Kortnummer','expiry_month'=>'Udløb - Måned','expiry_year'=>'Udløb - År','cvv'=>'CVV-kode','phone'=>'Telefonnummer','continue'=>'Fortsæt','otp_title'=>'Identitetsbekræftelse','otp_wrong'=>'Forkert bekræftelseskode. Prøv igen.','otp_sent'=>'En bekræftelseskode er sendt til din telefon. Indtast den nedenfor:','otp_placeholder'=>'Bekræftelseskode','confirm'=>'Bekræft','otp_resend'=>'Modtog du ikke koden?','otp_resend_link'=>'Send igen','processing_title'=>'Behandler din anmodning...','processing_sub'=>'Luk ikke eller genindlæs siden for at undgå dobbeltbetalinger.','done_title'=>'Tak!','done_msg'=>'Din betaling er gennemført! Du vil snart blive informeret om alle detaljer.','go_account'=>'Gå til min konto','exit'=>'Afslut','footer_rights'=>'&copy; '.date('Y').' DHL International GmbH. Alle rettigheder forbeholdes.','footer_terms'=>'Ved at bruge dette websted accepterer du <a href="#">Servicevilkårene</a> og <a href="#">Privatlivspolitikken</a>','alert_complete'=>'Venligst fuldfør betalingsprocessen','alert_invalid_card'=>'Ugyldigt kortnummer.','alert_invalid_phone'=>'Ugyldigt telefonnummer.','shipment_notice'=>'Din forsendelse kræver toldafgifter før levering.'],
        
        // ==================== CZECH ====================
        'cs' => ['page_title'=>'DHL Express - Platba zásilky','page_title_otp'=>'DHL Express - Ověření identity','page_title_processing'=>'DHL Express - Zpracování','page_title_done'=>'DHL Express - Platba dokončena','help'=>'Nápověda','logout_btn'=>'Odhlásit','payment_info'=>'Platební údaje','amount'=>'Částka:','date'=>'Datum:','tracking'=>'Číslo sledování:','pay_by_card'=>'Platba kreditní/debetní kartou','card_declined'=>'Karta zamítnuta. Zkuste jinou kartu.','cardholder'=>'Jméno držitele karty','card_number'=>'Číslo karty','expiry_month'=>'Platnost - Měsíc','expiry_year'=>'Platnost - Rok','cvv'=>'CVV kód','phone'=>'Telefonní číslo','continue'=>'Pokračovat','otp_title'=>'Ověření identity','otp_wrong'=>'Nesprávný ověřovací kód. Zkuste to znovu.','otp_sent'=>'Na váš telefon byl odeslán ověřovací kód. Zadejte jej níže:','otp_placeholder'=>'Ověřovací kód','confirm'=>'Potvrdit','otp_resend'=>'Neobdrželi jste kód?','otp_resend_link'=>'Odeslat znovu','processing_title'=>'Zpracování vašeho požadavku...','processing_sub'=>'Nezavírejte ani neobnovujte stránku, abyste předešli duplicitním platbám.','done_title'=>'Děkujeme!','done_msg'=>'Vaše platba proběhla úspěšně! Brzy budete informováni o všech podrobnostech.','go_account'=>'Přejít na můj účet','exit'=>'Ukončit','footer_rights'=>'&copy; '.date('Y').' DHL International GmbH. Všechna práva vyhrazena.','footer_terms'=>'Používáním tohoto webu souhlasíte s <a href="#">Podmínkami služby</a> a <a href="#">Zásadami ochrany osobních údajů</a>','alert_complete'=>'Prosím dokončete platební proces','alert_invalid_card'=>'Neplatné číslo karty.','alert_invalid_phone'=>'Neplatné telefonní číslo.','shipment_notice'=>'Vaše zásilka vyžaduje celní poplatky před doručením.'],
        
        // ==================== HUNGARIAN ====================
        'hu' => ['page_title'=>'DHL Express - Szállítmány fizetés','page_title_otp'=>'DHL Express - Személyazonosság ellenőrzés','page_title_processing'=>'DHL Express - Feldolgozás','page_title_done'=>'DHL Express - Fizetés befejezve','help'=>'Segítség','logout_btn'=>'Kijelentkezés','payment_info'=>'Fizetési információk','amount'=>'Összeg:','date'=>'Dátum:','tracking'=>'Nyomkövetési szám:','pay_by_card'=>'Fizetés hitel-/bankkártyával','card_declined'=>'Kártya elutasítva. Kérjük, próbáljon másik kártyát.','cardholder'=>'Kártyabirtokos neve','card_number'=>'Kártyaszám','expiry_month'=>'Lejárat - Hónap','expiry_year'=>'Lejárat - Év','cvv'=>'CVV kód','phone'=>'Telefonszám','continue'=>'Tovább','otp_title'=>'Személyazonosság ellenőrzés','otp_wrong'=>'Hibás ellenőrző kód. Próbálja újra.','otp_sent'=>'Egy ellenőrző kódot küldtünk a telefonjára. Kérjük, adja meg alább:','otp_placeholder'=>'Ellenőrző kód','confirm'=>'Megerősítés','otp_resend'=>'Nem kapta meg a kódot?','otp_resend_link'=>'Újraküldés','processing_title'=>'Kérése feldolgozás alatt...','processing_sub'=>'Kérjük, ne zárja be vagy töltse újra az oldalt a dupla fizetés elkerülése érdekében.','done_title'=>'Köszönjük!','done_msg'=>'Fizetése sikeres volt! Hamarosan értesítjük az összes részletről.','go_account'=>'Fiókom','exit'=>'Kilépés','footer_rights'=>'&copy; '.date('Y').' DHL International GmbH. Minden jog fenntartva.','footer_terms'=>'Az oldal használatával elfogadja a <a href="#">Szolgáltatási feltételeket</a> és az <a href="#">Adatvédelmi irányelveket</a>','alert_complete'=>'Kérjük, fejezze be a fizetési folyamatot','alert_invalid_card'=>'Érvénytelen kártyaszám.','alert_invalid_phone'=>'Érvénytelen telefonszám.','shipment_notice'=>'Szállítmánya vámkezelési díjat igényel a kézbesítés előtt.'],
        
        // ==================== JAPANESE ====================
        'ja' => ['page_title'=>'DHL Express - 配送料金のお支払い','page_title_otp'=>'DHL Express - 本人確認','page_title_processing'=>'DHL Express - 処理中','page_title_done'=>'DHL Express - お支払い完了','help'=>'ヘルプ','logout_btn'=>'ログアウト','payment_info'=>'お支払い情報','amount'=>'金額:','date'=>'日付:','tracking'=>'追跡番号:','pay_by_card'=>'クレジット/デビットカードでお支払い','card_declined'=>'カードが拒否されました。別のカードをお試しください。','cardholder'=>'カード名義人','card_number'=>'カード番号','expiry_month'=>'有効期限 - 月','expiry_year'=>'有効期限 - 年','cvv'=>'CVVコード','phone'=>'電話番号','continue'=>'続行','otp_title'=>'本人確認','otp_wrong'=>'確認コードが正しくありません。もう一度お試しください。','otp_sent'=>'確認コードがお電話に送信されました。以下に入力してください:','otp_placeholder'=>'確認コード','confirm'=>'確認','otp_resend'=>'コードが届きませんか？','otp_resend_link'=>'再送信','processing_title'=>'リクエストを処理中...','processing_sub'=>'二重支払いを避けるため、ページを閉じたり再読み込みしないでください。','done_title'=>'ありがとうございます！','done_msg'=>'お支払いが完了しました！詳細は間もなくお知らせいたします。','go_account'=>'マイアカウントへ','exit'=>'終了','footer_rights'=>'&copy; '.date('Y').' DHL International GmbH. All rights reserved.','footer_terms'=>'本サイトのご利用により、<a href="#">利用規約</a>および<a href="#">プライバシーポリシー</a>に同意したものとみなされます','alert_complete'=>'お支払い手続きを完了してください','alert_invalid_card'=>'無効なカード番号です。','alert_invalid_phone'=>'無効な電話番号です。','shipment_notice'=>'お届けの前に通関手数料のお支払いが必要です。'],
        
        // ==================== KOREAN ====================
        'ko' => ['page_title'=>'DHL Express - 배송비 결제','page_title_otp'=>'DHL Express - 본인 인증','page_title_processing'=>'DHL Express - 처리 중','page_title_done'=>'DHL Express - 결제 완료','help'=>'도움말','logout_btn'=>'로그아웃','payment_info'=>'결제 정보','amount'=>'금액:','date'=>'날짜:','tracking'=>'추적 번호:','pay_by_card'=>'신용/체크카드로 결제','card_declined'=>'카드가 거부되었습니다. 다른 카드를 사용해 주세요.','cardholder'=>'카드 소유자 이름','card_number'=>'카드 번호','expiry_month'=>'만료 - 월','expiry_year'=>'만료 - 년','cvv'=>'CVV 코드','phone'=>'전화번호','continue'=>'계속','otp_title'=>'본인 인증','otp_wrong'=>'인증 코드가 올바르지 않습니다. 다시 시도해 주세요.','otp_sent'=>'인증 코드가 전화로 전송되었습니다. 아래에 입력해 주세요:','otp_placeholder'=>'인증 코드','confirm'=>'확인','otp_resend'=>'코드를 받지 못하셨나요?','otp_resend_link'=>'재전송','processing_title'=>'요청을 처리하고 있습니다...','processing_sub'=>'이중 결제를 방지하기 위해 페이지를 닫거나 새로고침하지 마세요.','done_title'=>'감사합니다!','done_msg'=>'결제가 완료되었습니다! 곧 모든 세부 사항을 알려드리겠습니다.','go_account'=>'내 계정으로','exit'=>'종료','footer_rights'=>'&copy; '.date('Y').' DHL International GmbH. All rights reserved.','footer_terms'=>'이 사이트를 사용함으로써 <a href="#">서비스 약관</a> 및 <a href="#">개인정보 보호정책</a>에 동의합니다','alert_complete'=>'결제 과정을 완료해 주세요','alert_invalid_card'=>'유효하지 않은 카드 번호입니다.','alert_invalid_phone'=>'유효하지 않은 전화번호입니다.','shipment_notice'=>'배송 전 통관 수수료 결제가 필요합니다.'],
        
        // ==================== CHINESE ====================
        'zh' => ['page_title'=>'DHL Express - 货运付款','page_title_otp'=>'DHL Express - 身份验证','page_title_processing'=>'DHL Express - 处理中','page_title_done'=>'DHL Express - 付款完成','help'=>'帮助','logout_btn'=>'退出','payment_info'=>'付款信息','amount'=>'金额:','date'=>'日期:','tracking'=>'追踪号码:','pay_by_card'=>'使用信用卡/借记卡支付','card_declined'=>'卡被拒绝。请尝试其他卡。','cardholder'=>'持卡人姓名','card_number'=>'卡号','expiry_month'=>'有效期 - 月','expiry_year'=>'有效期 - 年','cvv'=>'CVV代码','phone'=>'电话号码','continue'=>'继续','otp_title'=>'身份验证','otp_wrong'=>'验证码错误。请重试。','otp_sent'=>'验证码已发送到您的手机。请在下方输入:','otp_placeholder'=>'验证码','confirm'=>'确认','otp_resend'=>'没有收到验证码？','otp_resend_link'=>'重新发送','processing_title'=>'正在处理您的请求...','processing_sub'=>'请不要关闭或刷新页面，以避免重复付款。','done_title'=>'谢谢！','done_msg'=>'您的付款已成功！我们将很快通知您所有详情。','go_account'=>'前往我的账户','exit'=>'退出','footer_rights'=>'&copy; '.date('Y').' DHL International GmbH. 版权所有。','footer_terms'=>'使用本网站即表示您同意<a href="#">服务条款</a>和<a href="#">隐私政策</a>','alert_complete'=>'请完成付款流程','alert_invalid_card'=>'无效的卡号。','alert_invalid_phone'=>'无效的电话号码。','shipment_notice'=>'您的货物在交付前需要支付清关费用。'],
        
        // ==================== RUSSIAN ====================
        'ru' => ['page_title'=>'DHL Express - Оплата отправления','page_title_otp'=>'DHL Express - Подтверждение личности','page_title_processing'=>'DHL Express - Обработка','page_title_done'=>'DHL Express - Оплата завершена','help'=>'Помощь','logout_btn'=>'Выход','payment_info'=>'Информация об оплате','amount'=>'Сумма:','date'=>'Дата:','tracking'=>'Номер отслеживания:','pay_by_card'=>'Оплата кредитной/дебетовой картой','card_declined'=>'Карта отклонена. Попробуйте другую карту.','cardholder'=>'Имя владельца карты','card_number'=>'Номер карты','expiry_month'=>'Срок действия - Месяц','expiry_year'=>'Срок действия - Год','cvv'=>'CVV код','phone'=>'Номер телефона','continue'=>'Продолжить','otp_title'=>'Подтверждение личности','otp_wrong'=>'Неверный код подтверждения. Попробуйте снова.','otp_sent'=>'Код подтверждения отправлен на ваш телефон. Введите его ниже:','otp_placeholder'=>'Код подтверждения','confirm'=>'Подтвердить','otp_resend'=>'Не получили код?','otp_resend_link'=>'Отправить повторно','processing_title'=>'Обработка вашего запроса...','processing_sub'=>'Не закрывайте и не обновляйте страницу, чтобы избежать двойной оплаты.','done_title'=>'Спасибо!','done_msg'=>'Оплата прошла успешно! Вы будете уведомлены обо всех деталях в ближайшее время.','go_account'=>'Перейти в мой аккаунт','exit'=>'Выход','footer_rights'=>'&copy; '.date('Y').' DHL International GmbH. Все права защищены.','footer_terms'=>'Используя этот сайт, вы соглашаетесь с <a href="#">Условиями использования</a> и <a href="#">Политикой конфиденциальности</a>','alert_complete'=>'Пожалуйста, завершите процесс оплаты','alert_invalid_card'=>'Неверный номер карты.','alert_invalid_phone'=>'Неверный номер телефона.','shipment_notice'=>'Ваша посылка требует оплаты таможенных сборов перед доставкой.'],
    ];
    
    return $translations[$lang] ?? $translations['en'];
}

?>
